<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingRules\Model;

use Magento\Checkout\Model\Session;
use Magento\Framework\App\Request\Http;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Magento\Quote\Api\Data\AddressInterface;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Address;
use Magento\Quote\Model\Quote\Address\Rate;
use Magento\Quote\Model\Quote\Address\RateResult\Method;
use Magento\Quote\Model\Quote\Item\AbstractItem as QuoteItem;
use Magento\Quote\Model\QuoteRepository;
use Magento\SalesRule\Model\Rule\Condition\Product;
use Magento\SalesRule\Model\Rule\Condition\Product\Combine;
use Magento\Webapi\Controller\Rest\InputParamsResolver;
use MageWorx\ShippingRules\Api\QuoteSessionManagerInterface;
use MageWorx\ShippingRules\Model\ResourceModel\Rule\Collection;
use MageWorx\ShippingRules\Model\ResourceModel\Rule\CollectionFactory;

/**
 * ShippingRules Validator Model
 *
 * @method mixed getStoreId()
 * @method Validator setStoreId($id)
 * @method mixed getCustomerGroupId()
 * @method Validator setCustomerGroupId($id)
 */
class Validator extends AbstractModel
{
    /**
     * Rule source collection
     *
     * @var Collection
     */
    protected $rules;

    /**
     * @var CollectionFactory
     */
    protected $collectionFactory;

    /**
     * @var Utility
     */
    protected $validatorUtility;

    /**
     * @var Session|\Magento\Backend\Model\Session\Quote
     */
    protected $session;

    /**
     * @var Product
     */
    protected $productCondition;

    protected $appliedShippingRuleIds  = [];
    protected $disabledShippingMethods = [];

    /**
     * @var QuoteRepository
     */
    protected $quoteRepository;

    /**
     * @var Quote
     */
    protected $quote;

    /**
     * @var Http
     */
    private $request;

    /**
     * @var Address
     */
    private $actualShippingAddress;

    /**
     * @var InputParamsResolver
     */
    private $inputParamsResolver;

    /**
     * @param Context $context
     * @param Registry $registry
     * @param CollectionFactory $collectionFactory
     * @param Utility $utility
     * @param QuoteSessionManagerInterface $quoteSessionManager
     * @param Product $productCondition
     * @param QuoteRepository $quoteRepository
     * @param Http $request
     * @param AbstractResource|null $resource
     * @param AbstractDb|null $resourceCollection
     * @param array $data
     */
    public function __construct(
        Context                      $context,
        Registry                     $registry,
        CollectionFactory            $collectionFactory,
        Utility                      $utility,
        QuoteSessionManagerInterface $quoteSessionManager,
        Product                      $productCondition,
        QuoteRepository              $quoteRepository,
        Http                         $request,
        InputParamsResolver          $inputParamsResolver,
        ?AbstractResource            $resource = null,
        ?AbstractDb                  $resourceCollection = null,
        array                        $data = []
    ) {
        $this->collectionFactory   = $collectionFactory;
        $this->validatorUtility    = $utility;
        $this->productCondition    = $productCondition;
        $this->session             = $quoteSessionManager->getActualSession();
        $this->quoteRepository     = $quoteRepository;
        $this->request             = $request;
        $this->inputParamsResolver = $inputParamsResolver;
        parent::__construct($context, $registry, $resource, $resourceCollection, $data);
    }

    /**
     * Init validator
     * Init process load collection of rules for specific store and
     * customer group
     *
     * @param int $storeId
     * @param int $customerGroupId
     * @return $this
     */
    public function init($storeId, $customerGroupId)
    {
        if ($this->quote && $this->quote->getCustomerGroupId() && !$customerGroupId) {
            $customerGroupId = $this->quote->getCustomerGroupId();
        }

        $this->setStoreId($storeId)->setCustomerGroupId($customerGroupId);

        $key = $storeId . '_' . $customerGroupId;
        if (!isset($this->rules[$key])) {
            /** @var Collection $collection */
            $collection        = $this->collectionFactory->create()
                                                         ->setValidationFilter(
                                                             $storeId,
                                                             $customerGroupId
                                                         )
                                                         ->addFieldToFilter('is_active', 1);
            $this->rules[$key] = $collection->load();
        }

        return $this;
    }

    /**
     * Get available stored rules for $rate
     *
     * @param Rate|Method $rate
     * @return array
     */
    public function getAvailableRulesForRate($rate)
    {
        /** @var string $currentMethod */
        $currentMethod = Rule::getMethodCode($rate);

        if (isset($this->appliedShippingRuleIds[$currentMethod]) &&
            count($this->appliedShippingRuleIds[$currentMethod])
        ) {
            return $this->appliedShippingRuleIds[$currentMethod];
        }

        return [];
    }

    /**
     * Check is item valid for the corresponding rule
     *
     * @param Rule $rule
     * @param QuoteItem $item
     * @return bool
     */
    public function isValidItem(Rule $rule, QuoteItem $item)
    {
        /** @var Combine $actions */
        $actions = $rule->getActions();
        if (!$actions->validate($item)) {
            return false;
        }

        return true;
    }

    /**
     * @param Rate|Method $rate
     * @return bool
     */
    public function validate($rate)
    {
        try {
            $address = $this->resolveShippingAddress();
        } catch (LocalizedException $e) {
            $this->_logger->critical($e);
            return false;
        }

        /** @var array $quoteItems */
        $quoteItems = $address->getAllItems() ? $address->getAllItems() : $this->quote->getAllItems();

        /** @var string $currentMethod */
        $currentMethod = Rule::getMethodCode($rate);

        // Do not process request without items (unusual request)
        if (empty($quoteItems)) {
            return false;
        }

        $this->_eventManager->dispatch(
            'mwx_start_rules_validation_processing',
            [
                'log_type'       => 'startRulesValidationProcessing',
                'current_method' => $currentMethod,
            ]
        );

        /* @var Rule $rule */
        foreach ($this->_getRules() as $rule) {
            // If rule has been already applied - continue
            if (isset($this->appliedShippingRuleIds[$currentMethod][$rule->getId()]) ||
                $this->checkAddressAppliedRule($address, $rule, $currentMethod)
            ) {
                if ($rule->getStopRulesProcessing()) {
                    break;
                }
                continue;
            }

            $this->_eventManager->dispatch(
                'mwx_start_validate_rule',
                [
                    'log_type' => 'startRuleValidation',
                    'rule'     => $rule,
                ]
            );

            // Validate rule conditions
            if (!$this->validatorUtility->canProcessRule($rule, $address, $currentMethod)) {
                $this->_eventManager->dispatch(
                    'mwx_invalid_rule',
                    [
                        'log_type' => 'logInvalidRule',
                        'rule'     => $rule,
                    ]
                );
                continue;
            }

            $this->appliedShippingRuleIds[$currentMethod][$rule->getId()] = $rule;
            $this->updateAddressAppliedShippingRuleIds($address);

            if ($rule->getStopRulesProcessing()) {
                $this->_eventManager->dispatch(
                    'mwx_rule_stop_processing',
                    [
                        'log_type' => 'logStopProcessingRule',
                        'rule'     => $rule,
                    ]
                );
                break;
            }

            $this->_eventManager->dispatch(
                'mwx_stop_validate_rule',
                [
                    'log_type' => 'stopRuleValidation',
                    'rule'     => $rule,
                ]
            );
        }

        $this->_eventManager->dispatch(
            'mwx_stop_validate_all_rules',
            [
                'log_type'       => 'stopAllRulesValidation',
                'current_method' => $currentMethod,
            ]
        );

        $validationResult = isset($this->appliedShippingRuleIds[$currentMethod]) &&
            count($this->appliedShippingRuleIds[$currentMethod]);

        return $validationResult;
    }

    /**
     * Resolve actual shipping address for validation
     *
     * @return Address|\Magento\Customer\Model\Data\Address
     * @throws NoSuchEntityException
     * @throws LocalizedException
     */
    protected function resolveShippingAddress()
    {
        if ($this->getActualShippingAddress()) {
            return $this->getActualShippingAddress();
        }

        if ($this->quote === null) {
            /**
             * @important
             * When quote taken from session it is possible that shipping address country id (or another data) will be
             * rewritten by it's original info (when customer use address from the pre-saved collection on the checkout)
             */
            /** @var Quote $quote */
            try {
                $quoteIdFromSession = $this->session->getQuoteId();
                $quote              = $this->quoteRepository->getActive($quoteIdFromSession);
            } catch (NoSuchEntityException $e) {
                $this->_logger->debug(
                    __('Unable to get active quote with ID %1', (string)$quoteIdFromSession),
                    [
                        'quote_id'      => $quoteIdFromSession,
                        'quote_id_type' => gettype($quoteIdFromSession)
                    ]
                );
                $quote = $this->quote && $this->quote->getId() ? $this->quote : $this->session->getQuote();
                if (!$quote->getId() && $this->request->getParam('quote_id')) {
                    $quote = $this->quoteRepository->get($this->request->getParam('quote_id'));
                }
            }

            $this->quote = $quote;
        }

        /** @var Address $address */
        $address = $this->quote->getShippingAddress();
        $this->setActualShippingAddress($address);

        return $address;
    }

    /**
     * Get actual address (multi shipping)
     * @important Must be public to grant access from third-party plugins
     *
     * @return AddressInterface
     */
    public function getActualShippingAddress()
    {
        return $this->actualShippingAddress;
    }

    /**
     * Set actual address (multishipping)
     *
     * @param Address $address
     * @return $this
     */
    public function setActualShippingAddress($address): Validator
    {
        $this->actualShippingAddress = $address;

        return $this;
    }

    /**
     * Get rules collection for current object state
     *
     * @return Collection
     */
    protected function _getRules()
    {
        $key = $this->getStoreId() . '_' . $this->getCustomerGroupId();

        return $this->rules[$key];
    }

    /**
     * If rhe rule already has been applied to the address return true
     * else return false
     *
     * @param Address $address
     * @param Rule $rule
     * @param string $method
     * @return bool
     */
    protected function checkAddressAppliedRule(
        Address $address,
        Rule    $rule,
                $method
    ) {
        $appliedRules = $address->getAppliedShippingRulesIds();

        if (!is_array($appliedRules)) {
            return false;
        }

        if (empty($appliedRules[$rule->getId()])) {
            return false;
        }

        if (in_array($method, $appliedRules[$rule->getId()])) {
            return true;
        }

        return false;
    }

    /**
     * Update address applied rule ids with new rule id
     *
     * @param Address $address
     * @return Address
     */
    protected function updateAddressAppliedShippingRuleIds(
        Address $address
    ) {
        $addressRuleIds = $address->getAppliedShippingRulesIds();
        if (!$addressRuleIds) {
            $addressRuleIds = [];
        }

        $resultIds = array_merge($addressRuleIds, $this->appliedShippingRuleIds);
        $address->setAppliedShippingRulesIds($resultIds);

        return $address;
    }

    /**
     * Set actual quote for validation
     *
     * @param Quote $quote
     * @return Validator
     */
    public function setQuote(Quote $quote): Validator
    {
        $this->quote = $quote;

        return $this;
    }
}
