<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\DeliveryDate\Block\Product\View;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Registry;
use Magento\Framework\Serialize\Serializer\Json as SerializerJson;
use Magento\Framework\View\Element\Template;
use Magento\Framework\View\Element\Template\Context;
use MageWorx\DeliveryDate\Helper\Data as Helper;

/**
 * Class EstimatedDeliveryTime
 *
 * Block designed to display estimated delivery time on the product view page
 *
 */
class EstimatedDeliveryTime extends Template
{
    /**
     * @var Product
     */
    protected $_product = null;

    /**
     * Core registry
     *
     * @var Registry
     */
    protected $_coreRegistry = null;

    /**
     * @var Helper
     */
    protected $helper;

    /**
     * @var SerializerJson
     */
    private $serializer;

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * EstimatedDeliveryTime constructor.
     *
     * @param Context $context
     * @param Registry $registry
     * @param Helper $helper
     * @param SerializerJson $serializer
     * @param ProductRepositoryInterface $productRepository
     * @param array $data
     */
    public function __construct(
        Context $context,
        Registry $registry,
        Helper $helper,
        SerializerJson $serializer,
        ProductRepositoryInterface $productRepository,
        array $data = []
    ) {
        $this->_coreRegistry             = $registry;
        $this->helper                    = $helper;
        $this->serializer                = $serializer;
        $this->productRepository         = $productRepository;

        parent::__construct($context, $data);
    }

    /**
     * @return string
     */
    public function getJsLayout(): string
    {
        $this->processJsLayout();

        return $this->serializer->serialize($this->jsLayout);
    }

    /**
     * Add js layout for configurable products DD component
     */
    private function processJsLayout(): void
    {
        $this->jsLayout = $this->jsLayout ?? [];

        $product = $this->getProduct();
        if ($product &&
            $product->getTypeId() == \Magento\ConfigurableProduct\Model\Product\Type\Configurable::TYPE_CODE
        ) {
            $this->jsLayout['components']['delivery-date-info']['cpid'] = $product->getId();

            try {
                $childItems = $this->getProductChildItemsConfig($product);
            } catch (NoSuchEntityException $e) {
                $childItems = [];
            }
            $this->jsLayout['components']['delivery-date-info']['childItems'] = $childItems;

            $this->jsLayout['components']['delivery-date-info']['selectOptionsMessageEnabled'] =
                $this->helper->isSelectProductOptionsMessageVisible();
            $this->jsLayout['components']['delivery-date-info']['selectOptionsMessage']        =
                $this->helper->getSelectProductOptionsMessage();
            $this->jsLayout['components']['delivery-date-info']['errorMessage']                =
                $this->getErrorMessage();
            $this->jsLayout['components']['delivery-date-info']['displayErrorMessage']         =
                $this->isDisplayErrorMessage();
        }
    }

    /**
     * @param Product $product
     * @return array
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function getProductChildItemsConfig(\Magento\Catalog\Model\Product $product): array
    {
        $productTypeInstance = $product->getTypeInstance();
        $usedProducts        = $productTypeInstance->getUsedProducts($product);
        $config              = [];

        $attributes         = $productTypeInstance->getConfigurableAttributes($product);
        $superAttributeList = [];
        foreach ($attributes as $_attribute) {
            $attributeCode                                     = $_attribute->getProductAttribute()->getAttributeCode();
            $superAttributeList[$_attribute->getAttributeId()] = $attributeCode;
        }

        foreach ($usedProducts as $usedProduct) {
            $usedProduct = $this->productRepository->get(
                $usedProduct->getSku(),
                false,
                $product->getStoreId(),
                true
            );
            $id          = $usedProduct->getId();
            $config[$id] = [
                'super_attributes'      => [],
                'delivery_date_visible' => false,
                'delivery_date_message' => null
            ];

            foreach ($superAttributeList as $superAttrId => $superAttribute) {
                $config[$id]['super_attributes'][$superAttrId] = $usedProduct->getData($superAttribute);
            }

            $isVisible        = $this->isVisibleOnProduct($usedProduct);
            $isAllowed        = $this->isAllowedByProduct($usedProduct);

            if (!$isVisible) {
                $config[$id]['delivery_date_visible'] = false;
                continue;
            }

            if (!$isAllowed) {
                $config[$id]['delivery_date_visible'] = $this->isDisplayErrorMessage();
                $config[$id]['delivery_date_message'] = $this->getErrorMessage();
                continue;
            }

            $config[$id]['delivery_date_visible'] = true;
            $config[$id]['delivery_date_message'] = $this->getEstimatedDeliveryTime($usedProduct);
        }

        return $config;
    }

    /**
     * @return Product
     */
    public function getProduct()
    {
        if (!$this->_product) {
            $this->_product = $this->_coreRegistry->registry('product');
        }

        return $this->_product;
    }

    /**
     * Returns estimated delivery time message for product (based on the global configuration)
     *
     * @return string
     */
    public function getEstimatedDeliveryTime(\Magento\Catalog\Api\Data\ProductInterface $product = null)
    {
        $product  = $product ?? $this->getProduct();
        $daysFrom = (int)$product->getData('mw_delivery_time_from');
        $daysTo   = (int)$product->getData('mw_delivery_time_to');

        if ($daysFrom == 0) {
            if ($this->helper->getProductSameDayDeliveryMessage()) {
                $message = $this->helper->getProductSameDayDeliveryMessage();

                return $message;
            }
        } elseif ($daysFrom == 1) {
            if ($this->helper->getProductNextDayDeliveryMessage()) {
                $message = $this->helper->getProductNextDayDeliveryMessage();

                return $message;
            }
        }

        $format  = $this->helper->getProductEDTMessageFormat();
        $message = str_ireplace(Helper::PRODUCT_EDT_DAYS_FROM_NUMBER_VARIABLE, $daysFrom, $format);
        $message = str_ireplace(Helper::PRODUCT_EDT_DAYS_TO_NUMBER_VARIABLE, $daysTo, $message);

        return $message;
    }

    /**
     * Is display of Estimated Delivery Time enabled on the product
     *
     * @param \Magento\Catalog\Api\Data\ProductInterface|null $product
     * @return bool
     */
    public function isVisibleOnProduct(\Magento\Catalog\Api\Data\ProductInterface $product = null)
    {
        $product = $product ?? $this->getProduct();

        $visible = (bool)$product->getData('mw_delivery_time_visible');
        if (!$visible) {
            return false;
        }

        return $visible;
    }

    /**
     * Is Estimated Delivery Time enabled for the product
     *
     * @param \Magento\Catalog\Api\Data\ProductInterface|null $product
     * @return bool
     */
    public function isAllowedByProduct(\Magento\Catalog\Api\Data\ProductInterface $product = null)
    {
        $product = $product ?? $this->getProduct();

        $allowed = (bool)$product->getData('mw_delivery_time_enabled');
        if (!$allowed) {
            return false;
        }

        return $allowed;
    }

    /**
     * Is error message should be visible on front
     *
     * @return bool
     */
    public function isDisplayErrorMessage(): bool
    {
        return $this->helper->isErrorMessageVisibleOnProduct();
    }

    /**
     * Error message in case when Delivery Date unavailable right now
     *
     * @return string
     */
    public function getErrorMessage(): string
    {
        return $this->helper->getDeliveryDateProductErrorMessage();
    }
}
