<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\UrlRewrite\Model;

use Magento\Framework\App\ObjectManager;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\EntityManager\EventManager;
use Magento\Framework\Indexer\CacheContext;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\UrlRewrite\Controller\Adminhtml\Url\Rewrite;
use Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollection;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite as UrlRewriteService;

/**
 * UrlRewrite model class
 *
 * @method int getEntityId()
 * @method string getEntityType()
 * @method int getRedirectType()
 * @method int getStoreId()
 * @method int getIsAutogenerated()
 * @method string getTargetPath()
 * @method UrlRewrite setEntityId(int $value)
 * @method UrlRewrite setEntityType(string $value)
 * @method UrlRewrite setRequestPath($value)
 * @method UrlRewrite setTargetPath($value)
 * @method UrlRewrite setRedirectType($value)
 * @method UrlRewrite setStoreId($value)
 * @method UrlRewrite setDescription($value)
 * @api
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * @SuppressWarnings(PHPMD.ExcessiveParameterList)
 */
class UrlRewrite extends AbstractModel
{
    /**
     * @var Json
     */
    private $serializer;

    /**
     * @var CacheContext|mixed|null
     */
    private $cacheContext;

    /**
     * @var EventManager|mixed|null
     */
    private $eventManager;

    /**
     * @var array
     */
    private $entityToCacheTagMap;

    /**
     * @var UrlFinderInterface
     */
    private $urlFinder;

    /**
     * UrlRewrite constructor.
     *
     * @param Context $context
     * @param Registry $registry
     * @param AbstractResource|null $resource
     * @param AbstractDb|null $resourceCollection
     * @param array $data
     * @param Json|null $serializer
     * @param CacheContext|null $cacheContext
     * @param EventManager|null $eventManager
     * @param UrlFinderInterface|null $urlFinder
     * @param array $entityToCacheTagMap
     */
    public function __construct(
        Context $context,
        Registry $registry,
        ?AbstractResource $resource = null,
        ?AbstractDb $resourceCollection = null,
        array $data = [],
        ?Json $serializer = null,
        ?CacheContext $cacheContext = null,
        ?EventManager $eventManager = null,
        ?UrlFinderInterface $urlFinder = null,
        array $entityToCacheTagMap = []
    ) {
        $this->serializer = $serializer ?: ObjectManager::getInstance()->get(Json::class);
        $this->cacheContext = $cacheContext ?: ObjectManager::getInstance()->get(CacheContext::class);
        $this->eventManager = $eventManager ?: ObjectManager::getInstance()->get(EventManager::class);
        $this->urlFinder = $urlFinder ?: ObjectManager::getInstance()->get(UrlFinderInterface::class);
        $this->entityToCacheTagMap = $entityToCacheTagMap;
        parent::__construct($context, $registry, $resource, $resourceCollection, $data);
    }

    /**
     * Initialize corresponding resource model
     *
     * @return void
     */
    protected function _construct()
    {
        $this->_init(ResourceModel\UrlRewrite::class);
        $this->_collectionName = UrlRewriteCollection::class;
    }

    /**
     * Get metadata
     *
     * @return array
     */
    public function getMetadata()
    {
        $metadata = $this->getData(\Magento\UrlRewrite\Service\V1\Data\UrlRewrite::METADATA);
        return !empty($metadata) ? $this->serializer->unserialize($metadata) : [];
    }

    /**
     * Overwrite Metadata in the object.
     *
     * @param array|string $metadata
     * @return $this
     */
    public function setMetadata($metadata)
    {
        if (is_array($metadata)) {
            $metadata = $this->serializer->serialize($metadata);
        }
        return $this->setData(\Magento\UrlRewrite\Service\V1\Data\UrlRewrite::METADATA, $metadata);
    }

    /**
     * Gets final target UrlRewrite for custom rewrite record
     *
     * @param string $path
     * @param int $storeId
     * @return UrlRewriteService|null
     */
    private function getFinalTargetUrlRewrite(string $path, int $storeId): ?UrlRewriteService
    {
        $urlRewriteTarget = $this->urlFinder->findOneByData(
            [
                'request_path' => $path,
                'store_id' => $storeId
            ]
        );

        // to manage accent characters in URL rewrite
        if ($urlRewriteTarget) {
            $planeChars = iconv('UTF-8', 'ISO-8859-1//IGNORE', $urlRewriteTarget->getRequestPath());

            if ($planeChars !== $urlRewriteTarget->getRequestPath()) {
                $urlRewriteTarget = null;
            }
        }

        while ($urlRewriteTarget &&
            $urlRewriteTarget->getTargetPath() !== $urlRewriteTarget->getRequestPath() &&
            $urlRewriteTarget->getRedirectType() > 0
        ) {
            $urlRewriteTarget = $this->urlFinder->findOneByData(
                [
                    'request_path' => $urlRewriteTarget->getTargetPath(),
                    'store_id' => $urlRewriteTarget->getStoreId()
                ]
            );
        }

        return $urlRewriteTarget;
    }

    /**
     * Clean the cache for entities affected by current rewrite
     */
    public function cleanEntitiesCache()
    {
        if (!$this->isEmpty()) {
            if ($this->getEntityType() === Rewrite::ENTITY_TYPE_CUSTOM) {
                $urlRewrite = $this->getFinalTargetUrlRewrite(
                    $this->getTargetPath(),
                    (int)$this->getStoreId()
                );

                if ($urlRewrite) {
                    $this->cleanCacheForEntity($urlRewrite->getEntityType(), (int) $urlRewrite->getEntityId());
                }

                if ($this->getOrigData() && $this->getOrigData('target_path') !== $this->getTargetPath()) {
                    $origUrlRewrite = $this->getFinalTargetUrlRewrite(
                        $this->getOrigData('target_path'),
                        (int)$this->getOrigData('store_id')
                    );

                    if ($origUrlRewrite) {
                        $this->cleanCacheForEntity(
                            $origUrlRewrite->getEntityType(),
                            (int) $origUrlRewrite->getEntityId()
                        );
                    }
                }
            } else {
                $this->cleanCacheForEntity($this->getEntityType(), (int) $this->getEntityId());
            }
        }
    }

    /**
     * Clean cache for specified entity type by id
     *
     * @param string $entityType
     * @param int $entityId
     */
    private function cleanCacheForEntity(string $entityType, int $entityId)
    {
        if (array_key_exists($entityType, $this->entityToCacheTagMap)) {
            $cacheKey = $this->entityToCacheTagMap[$entityType];
            $this->cacheContext->registerEntities($cacheKey, [$entityId]);
            $this->eventManager->dispatch('clean_cache_by_tags', ['object' => $this->cacheContext]);
        }
    }

    /**
     * @inheritdoc
     */
    public function afterDelete()
    {
        $this->_getResource()->addCommitCallback([$this, 'cleanEntitiesCache']);
        return parent::afterDelete();
    }

    /**
     * @inheritdoc
     */
    public function afterSave()
    {
        $this->_getResource()->addCommitCallback([$this, 'cleanEntitiesCache']);
        return parent::afterSave();
    }
}
