<?php
/**
 * Mageplaza
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Mageplaza.com license that is
 * available through the world-wide-web at this URL:
 * https://www.mageplaza.com/LICENSE.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category  Mageplaza
 * @package   Mageplaza_BetterProductReviews
 * @copyright Copyright (c) Mageplaza (https://www.mageplaza.com/)
 * @license   https://www.mageplaza.com/LICENSE.txt
 */

namespace Mageplaza\CustomForm\Controller\ViewFile;

use Exception;
use Psr\Log\LoggerInterface;
use Magento\Framework\Filesystem;
use Mageplaza\CustomForm\Helper\Data;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Mageplaza\CustomForm\Helper\FileType;
use Magento\Framework\View\Element\Template;
use Magento\Framework\Controller\Result\Json;
use Magento\Framework\View\Result\PageFactory;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Filesystem\Directory\Read;
use Laminas\Validator\File\Upload as FileUpload;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Exception\NotFoundException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\MediaStorage\Model\File\UploaderFactory;

class HyvaUpload extends Action
{
    /**
     * @var PageFactory
     */
    protected PageFactory $pageFactory;

    /**
     * @var FileUpload
     */
    protected FileUpload $fileUpload;

    /**
     * @var LoggerInterface
     */
    protected LoggerInterface $logger;

    /**
     * @var UploaderFactory
     */
    protected UploaderFactory $uploaderFactory;

    /**
     * @var Filesystem
     */
    protected Filesystem $filesystem;

    /**
     * @var Data
     */
    protected Data $data;

    /**
     * @var FileType
     */
    protected FileType $fileTypeHelper;

    /**
     * @param Context $context
     * @param PageFactory $pageFactory
     * @param FileUpload $fileUpload
     * @param LoggerInterface $logger
     * @param UploaderFactory $uploaderFactory
     * @param Filesystem $filesystem
     * @param Data $data
     * @param FileType $fileTypeHelper
     */
    public function __construct(
        Context $context,
        PageFactory $pageFactory,
        FileUpload $fileUpload,
        LoggerInterface $logger,
        UploaderFactory $uploaderFactory,
        Filesystem $filesystem,
        Data $data,
        FileType $fileTypeHelper
    ) {
        $this->pageFactory     = $pageFactory;
        $this->fileUpload      = $fileUpload;
        $this->logger          = $logger;
        $this->uploaderFactory = $uploaderFactory;
        $this->filesystem      = $filesystem;
        $this->data            = $data;
        $this->fileTypeHelper  = $fileTypeHelper;
        parent::__construct($context);
    }

    /**
     * Execute action based on request and return result
     *
     * @return ResultInterface|ResponseInterface
     * @throws NotFoundException
     */
    public function execute()
    {
        try {
            $page         = $this->pageFactory->create();
            $layout       = $page->getLayout();
            $uploadedFile = $this->fileUpload->getFiles();
            if (empty($uploadedFile)) {
                throw new LocalizedException(__('$_FILES array is empty.'));
            }

            $allowedMime       = [];
            $allowedExtensions = $this->getRequest()->getParam('allowed_extensions');
            $allowedExtensions = $allowedExtensions ? explode(',', $allowedExtensions ?? '') : [];
            if (!empty($allowedExtensions)) {
                $mimeType  = $uploadedFile['file']['type'];
                $mimeTypes = $this->fileTypeHelper->getMimeTypes();
                foreach ($allowedExtensions as $type) {
                    $allowedMime[] = $mimeTypes[$type];
                }
                if (!in_array($mimeType, $allowedMime, true)) {
                    throw new LocalizedException(__('We don\'t recognize or support this file extension type.'));
                }
            }

            $maxFileSize = $this->getRequest()->getParam('max_file_size');
            if ($maxFileSize && $maxFileSize < $uploadedFile['file']['size']) {
                throw new LocalizedException(__(sprintf('The file size must be smaller than %s.', $maxFileSize)));
            }

            $fileId   = key($uploadedFile);
            $uploader = $this->uploaderFactory->create(['fileId' => $fileId]);
            $uploader->setAllowRenameFiles(true);
            $uploader->setFilesDispersion(true);

            /** @var Read $mediaDirectory */
            $mediaDirectory = $this->filesystem->getDirectoryRead(DirectoryList::MEDIA);

            $now    = time();
            $result = $uploader->save(
                $mediaDirectory->getAbsolutePath(\Mageplaza\CustomForm\Helper\Data::FILE_MEDIA_PATH) . '/' . $now
            );
            unset($result['path'], $result['tmp_name']);
            $result['file'] = $now . '/' . $result['file'];
            $result['url']  = $this->data->getTmpMediaUrl($result['file']);

            $data = [
                'label'          => $result['name'],
                'file'           => $result['file'],
                'url'            => $result['url'],
                'type'           => $this->getFilePreviewType($result),
                'formatted_size' => $this->formatSize($result['size']),
            ];

            if ($data['type'] == 'image' && $imageData = file_get_contents($data['url'])) {
                list($width, $height) = getimagesizefromstring($imageData);
                $data['previewWidth'] = $width;
                $data['previewHeight'] = $height;
            }

            $response = [
                'previewer' => $layout->createBlock(Template::class)
                    ->setTemplate('Mageplaza_CustomForm::hyva/review/upload/file.phtml')
                    ->setFileData($data)
                    ->toHtml(),
                'file'      => $data['file'],
                'success'   => true,
            ];
        } catch (Exception $e) {
            $this->logger->critical($e);
            $response = [
                'success'   => false,
                'error'     => __($e->getMessage()),
                'errorcode' => $e->getCode(),
            ];
        }

        /** @var Json $resultJson */
        $resultJson = $this->resultFactory->create(ResultFactory::TYPE_JSON);
        $resultJson->setData($response);

        return $resultJson;
    }

    /**
     * @param $result
     *
     * @return string
     */
    public function getFilePreviewType($result): string
    {
        if (empty($result['type'])) {
            return 'document';
        }

        $type = explode('/', $result['type'])[0];

        return ($type !== 'image' && $type !== 'video') ? 'document' : $type;
    }

    /**
     * @param $bytes
     *
     * @return string
     */
    public function formatSize($bytes): string
    {
        $sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
        if ($bytes === 0) {
            return '0 Byte';
        }

        $i = (int) floor(log($bytes) / log(1024));

        return round($bytes / pow(1024, $i), 2) . ' ' . $sizes[$i];
    }
}
