/**
 * Copyright © MageWorx, Inc. All rights reserved.
 * See LICENSE.txt for license details.
 */
define(
    [
        'jquery',
        'Magento_Ui/js/form/form',
        'uiRegistry',
        'mageUtils',
        'ko',
        'mage/validation'
    ],
    function (
        $,
        Component,
        registry,
        utils,
        ko
    ) {
        'use strict';

        /**
         * Format params
         *
         * @param {Object} params
         * @returns {Array}
         */
        function prepareParams(params) {
            var result = '?';

            _.each(params, function (value, key) {
                result += key + '=' + value + '&';
            });

            return result.slice(0, -1);
        }

        /**
         * Collect form data.
         *
         * @param {Array} items
         * @returns {Object}
         */
        function collectData(items) {
            var result = {},
                name;

            items = Array.prototype.slice.call(items);

            items.forEach(function (item) {
                switch (item.type) {
                    case 'checkbox':
                        result[item.name] = +!!item.checked;
                        break;

                    case 'radio':
                        if (item.checked) {
                            result[item.name] = item.value;
                        }
                        break;

                    case 'select-multiple':
                        name = item.name.substring(0, item.name.length - 2); //remove [] from the name ending
                        result[name] = _.pluck(item.selectedOptions, 'value');
                        break;

                    default:
                        result[item.name] = item.value;
                }
            });

            return result;
        }

        /**
         * Makes ajax request
         *
         * @param {Object} params
         * @param {Object} data
         * @param {String} url
         * @param {boolean} loader
         * @returns {*}
         */
        function makeRequest(params, data, url, skipLoader) {
            var save = $.Deferred();

            data = utils.serialize(data);
            data['form_key'] = window.FORM_KEY;

            if (!url) {
                save.resolve();
            }

            if (!skipLoader) {
                $('body').trigger('processStart');
            }

            $.ajax({
                url: url + prepareParams(params),
                data: data,
                dataType: 'json',

                /**
                 * Success callback.
                 * @param {Object} resp
                 * @returns {Boolean}
                 */
                success: function (resp) {
                    if (resp.ajaxExpired) {
                        window.location.href = resp.ajaxRedirect;
                    }

                    if (!resp.criticalError) {
                        save.resolve(resp);

                        return true;
                    }

                    $('body').notification('clear');
                    $.each(resp.messages, function (key, message) {
                        $('body').notification('add', {
                            error: resp.error,
                            message: message,

                            /**
                             * Inserts message on page
                             * @param {String} msg
                             */
                            insertMethod: function (msg) {
                                $('.page-main-actions').after(msg);
                            }
                        });
                    });
                },

                /**
                 * Complete callback.
                 */
                complete: function () {
                    if (!skipLoader) {
                        $('body').trigger('processStop');
                    }
                }
            });

            return save.promise();
        }

        return Component.extend({
            defaults: {
                template: 'MageWorx_ShippingCalculatorBase/view/form',
                calculateShippingMethodsOnLoad: false,
                imports: {
                    country: 'shipping-calculator.estimateForm.fieldset.country_id:value',
                    region_id: 'shipping-calculator.estimateForm.fieldset.region_id:value',
                    region: 'shipping-calculator.estimateForm.fieldset.region:value',
                    postcode: 'shipping-calculator.estimateForm.fieldset.postcode:value',
                    regionOptions: 'shipping-calculator.estimateForm.fieldset.region_id:options',
                    countryOptions: 'shipping-calculator.estimateForm.fieldset.country_id:options'
                },
                exports: {
                    methods: 'shipping-calculator.methods-list:methods'
                }
            },

            /**
             * Properties which will be set as an observable during initObservable method call
             */
            observableProperties: [
                'refreshUrl',
                'visible',
                'defaultLabel',
                'country',
                'region',
                'region_id',
                'postcode',
                'regionOptions',
                'countryOptions',
                'regionTitle',
                'methods',
                'calculateShippingMethodsOnLoad'
            ],

            /**
             * @override
             */
            initialize: function () {
                this._super();

                return this;
            },

            /**
             * Initialize observables properties
             *
             * @returns {exports}
             */
            initObservable: function () {
                this._super();
                this.observe(this.observableProperties);

                this.addressFormatted('Belarus, Minsk, 220034');

                return this;
            },

            addressFormatted: function () {
                var self = this;
                return ko.computed(function () {
                    var country = self.country(),
                        region,
                        zip = self.postcode(),
                        result = '',
                        regionOptions = self.regionOptions(),
                        countryOptions = self.countryOptions();

                    if (self.region_id()
                        && regionOptions
                    ) {
                        var desiredRegionConfig = regionOptions.filter(function(e) {
                            return e['value'] && (e['value'] == self.region_id());
                        });
                        if (desiredRegionConfig[0]) {
                            region = desiredRegionConfig[0].label;
                        } else {
                            region = self.region_id();
                        }
                    } else if (self.region() && (!regionOptions || !regionOptions.length)) {
                        region = self.region();
                    } else if (self.region_id()) {
                        region = self.region_id();
                    }

                    if (country) {
                        if (countryOptions) {
                            var countryOption = countryOptions.filter(function (e) {
                                return e['value'] && (e['value'] == country);
                            });
                            if (countryOption && countryOption.length) {
                                result += countryOption[0].label;
                            } else {
                                result += country;
                            }
                        } else {
                            result += country;
                        }
                    }

                    if (region) {
                        result += (country ? ', ' : '') + region;
                    }

                    if (zip) {
                        result += (country || region ? ', ' : '') + zip;
                    }

                    return result;
                })();
            },

            estimateShippingMethodsAfterLoad: function () {
                if (!this.calculateShippingMethodsOnLoad()) {
                    return;
                }

                this.estimateShippingMethods(true, true);
            },

            /**
             * Send form data to a server and accept an available shipping methods
             */
            estimateShippingMethods: function (skipLoader, skipValidation) {
                var self = this,
                    $productForm = $('#product_addtocart_form');

                if ($productForm.validation('isValid') && (skipValidation || this.validate())) {
                    var methodsList = registry.get('index = methods-list');
                    if (methodsList) {
                        methodsList.productFormChanged(false);
                    }

                    makeRequest(this.params, this.collectData(), this.refreshUrl(), skipLoader).then(function (data) {
                        var calculator = self.getCalculator();
                        calculator.clearErrors();
                        self.clearMethodsData();
                        if (data.success) {
                            if (data.shipping_methods) {
                                self.addMethodsData(data.shipping_methods);
                            }
                        }

                        if (data.error) {
                            calculator.addErrorMessage(data.error);
                        }
                    });
                }
            },

            /**
             * Validate address form
             *
             * @returns {boolean}
             */
            validate: function ()
            {
                this._super();

                var items = registry.get('dataScope = shippingAddress, index = fieldset').elems(),
                    result = true,
                    source = this.source;

                _.each (items, function (item) {
                    if (!item.skipValidation && typeof item.validate === 'function') {
                        item.validate();
                        if (source.get('params.invalid') !== false) {
                            result = false;
                        }
                    }
                });

                return result;
            },

            /**
             * Remove existing methods data
             */
            clearMethodsData: function () {
                var methodsList = registry.get('index = methods-list');
                if (methodsList) {
                    methodsList.destroyChildren();
                }
            },

            /**
             * Generate new methods data records on front
             *
             * @param methods
             */
            addMethodsData: function (methods)
            {
                this.methods(methods);
                if (methods && methods.length > 0) {
                    this.visible(false);
                }
            },

            /**
             * Collect all data for request
             * @returns {{}}
             */
            collectData: function () {
                var data = {};
                data.address = this.source.shippingAddress;
                data.item = this.getProductData();

                return data;
            },

            /**
             * Collect data about product (potential buyRequest)
             * @returns {{}}
             */
            getProductData: function () {
                var $form = $('#product_addtocart_form'),
                    data = {};

                if ($form.length > 0) {
                    data = $form.serializeArray();
                }

                return data;
            },

            /**
             * Returns the shipping calculator instance from registry
             *
             * @returns {Object}
             */
            getCalculator: function () {
                return registry.get('index = shipping-calculator');
            }
        });
    }
);
