<?php
/**
 * Copyright © 2016 MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\OptionLink\Plugin;

use Magento\Catalog\Model\ResourceModel\Product\Option\Value\Collection as ProductOptionValueCollection;
use Magento\Framework\App\State;
use Magento\Framework\ObjectManagerInterface as ObjectManager;
use Magento\Framework\Registry;
use MageWorx\OptionLink\Helper\Attribute as HelperAttribute;
use MageWorx\OptionLink\Model\ResourceModel\Product\Option\Value\CollectionUpdater;
use MageWorx\OptionLink\Model\ResourceModel\Product\Option\Value\FieldFactory;
use Psr\Log\LoggerInterface;

/**
 * Class BeforeLoad. Replace option fields data to product attributes selected in setting.
 * This class are used when Option Value Collection are loading.
 */
class BeforeLoad
{
    protected HelperAttribute   $helperAttribute;
    protected FieldFactory      $optionFieldFactory;
    protected CollectionUpdater $collectionUpdater;
    protected ?ObjectManager    $objectManager = null;
    protected Registry          $registry;
    protected LoggerInterface   $logger;
    protected State             $appState;

    /**
     * BeforeLoad constructor.
     *
     * @param HelperAttribute $helperAttribute
     * @param FieldFactory $optionFieldFactory
     * @param CollectionUpdater $collectionUpdater
     * @param ObjectManager $objectManager
     * @param Registry $registry
     */
    public function __construct(
        HelperAttribute   $helperAttribute,
        FieldFactory      $optionFieldFactory,
        CollectionUpdater $collectionUpdater,
        ObjectManager     $objectManager,
        Registry          $registry,
        State             $appState,
        LoggerInterface   $logger
    ) {
        $this->helperAttribute    = $helperAttribute;
        $this->optionFieldFactory = $optionFieldFactory;
        $this->collectionUpdater  = $collectionUpdater;
        $this->objectManager      = $objectManager;
        $this->registry           = $registry;
        $this->appState           = $appState;
        $this->logger             = $logger;
    }

    /**
     * @param ProductOptionValueCollection $collection
     * @param bool $printQuery
     * @param bool $logQuery
     * @return array
     */
    public function beforeLoad($collection, $printQuery = false, $logQuery = false)
    {
        $isOptionTemplateSave = $this->registry->registry('mageworx_optiontemplates_group_save');
        if ($isOptionTemplateSave) {
            return [$printQuery, $logQuery];
        }

        $fields = $this->helperAttribute->getConvertedAttributesToFields();

        if (!$this->canUpdate($collection, $fields)) {
            return [$printQuery, $logQuery];
        }

        $this->collectionUpdater->joinProductTable($collection);

        $this->collectionUpdater->resetColumns($collection, $fields);

        $this->collectionUpdater->addOriginalColumns($collection, $fields);

        foreach ($fields as $field) {
            if ($this->collectionUpdater->canAddField($collection, $field)) {
                $this->optionFieldFactory->create($field)->addField($collection);
            }
        }

        return [$printQuery, $logQuery];
    }

    /**
     * Check if we can modify collection sql.
     *
     * @param $selectFields
     * @param $productAttributes
     * @return bool
     */
    protected function canUpdate(ProductOptionValueCollection $collection, array $fields): bool
    {
        // We modify sql when selected 1 or more product attributes.
        if (empty($fields)) {
            return false;
        }

        // Return False if sql was modified
        $sqlFrom = $collection->getSelect()->getPart('from');
        if (in_array(CollectionUpdater::KEY_TABLE_OPTIONLINK_PRODUCT, array_keys($sqlFrom))) {
            return false;
        }

        // Check count fields in "select" part of sql.
        // We modify sql which contains 2 or more fields.
        $sqlColumns = $collection->getSelect()->getPart('columns');
        if (count($sqlColumns) <= 1) {
            return false;
        }

        // Check the sku_is_valid column and return false in case when no value is eq 1. Only for frontend requests.
        if ($this->isFrontendRequest() && !$this->hasAtLeastOneValidSku($collection)) {
            return false;
        }

        return true;
    }

    /**
     * Check if at least one sku is valid.
     *
     * @param ProductOptionValueCollection $collection
     * @return bool
     * @throws \Zend_Db_Select_Exception
     */
    protected function hasAtLeastOneValidSku(ProductOptionValueCollection $collection): bool
    {
        try {
            $where              = $collection->getSelect()->getPart('where');
            $optionIdsCondition = $this->extractOptionIdsFromWhere($where);

            if (!$optionIdsCondition) {
                return false;
            }

            $connection = $collection->getConnection();
            $mainTable  = $collection->getResource()->getMainTable();
            $select     = $connection->select()
                                     ->from(['main_table' => $mainTable], ['sku_is_valid'])
                                     ->where($optionIdsCondition)
                                     ->where('main_table.disabled = ?', 0)
                                     ->order('sku_is_valid DESC')
                                     ->limit(1);

            $result = $connection->fetchOne($select);

            return (int)$result === 1;
        } catch (\Exception $e) {
            $this->logger->warning('SKU check failed in OptionLink plugin: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * @param array $whereParts
     * @return string
     */
    protected function extractOptionIdsFromWhere(array $whereParts): string
    {
        foreach ($whereParts as $part) {
            if (stripos($part, 'option_id IN') !== false) {
                $start           = stripos($part, 'option_id');
                $substring       = substr($part, $start);
                $openBracketPos  = strpos($substring, '(');
                $closeBracketPos = strrpos($substring, ')');

                if ($openBracketPos !== false && $closeBracketPos !== false) {
                    $inClause = substr($substring, 0, $closeBracketPos + 1);
                    return $inClause;
                }
            }
        }

        return '';
    }

    /**
     * @return bool
     */
    protected function isFrontendRequest(): bool
    {
        try {
            return $this->appState->getAreaCode() === \Magento\Framework\App\Area::AREA_FRONTEND;
        } catch (\Exception $e) {
            return false;
        }
    }
}
