<?php
namespace WeltPixel\Sitemap\Model\ResourceModel\Catalog;

use Magento\CatalogUrlRewrite\Model\CategoryUrlRewriteGenerator;
use Magento\Framework\App\ObjectManager;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\Store;

/**
 * Class Category
 * @package WeltPixel\Sitemap\Model\ResourceModel\Catalog
 */
class Category extends \Magento\Sitemap\Model\ResourceModel\Catalog\Category
{

    /**
     * Scope Config
     *
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    private $scopeConfig;


    /**
     * Get category collection array
     *
     * @param null|string|bool|int|\Magento\Store\Model\Store $storeId
     * @return array|bool
     */
    public function getCollection($storeId)
    {
        $categories = [];

        /* @var $store \Magento\Store\Model\Store */
        $store = $this->_storeManager->getStore($storeId);

        if (!$store) {
            return false;
        }

        $useCanonicalUrlForCategory = $this->isCanonicalUrlForCategory($storeId);
        $connection = $this->getConnection();

        $this->_select = $connection->select()->from(
            $this->getMainTable()
        )->where(
            $this->getIdFieldName() . '=?',
            $store->getRootCategoryId()
        );
        $categoryRow = $connection->fetchRow($this->_select);

        if (!$categoryRow) {
            return false;
        }

        $this->_select = $connection->select()->from(
            ['e' => $this->getMainTable()],
            [$this->getIdFieldName(), 'updated_at']
        )->joinLeft(
            ['url_rewrite' => $this->getTable('url_rewrite')],
            'e.entity_id = url_rewrite.entity_id AND url_rewrite.is_autogenerated = 1'
            . $connection->quoteInto(' AND url_rewrite.store_id = ?', $store->getId())
            . $connection->quoteInto(' AND url_rewrite.entity_type = ?', CategoryUrlRewriteGenerator::ENTITY_TYPE),
            ['url' => 'request_path']
        )->where(
            'e.path LIKE ?',
            $categoryRow['path'] . '/%'
        );

        $this->_addFilter($storeId, 'is_active', 1);
        $this->_addFilter($storeId, 'weltpixel_exclude_from_sitemap', 0);

        if ($useCanonicalUrlForCategory) {
            $this->_joinAttribute($store->getId(), 'wp_enable_canonical_url', 'wp_enable_canonical_url');
            $this->_joinAttribute($store->getId(), 'wp_canonical_url', 'wp_canonical_url');
        } else {
            $this->_joinAttribute($store->getId(), 'wp_use_canonical_url_in_sitemap', 'wp_use_canonical_url_in_sitemap');
            $this->_joinAttribute($store->getId(), 'wp_canonical_url', 'wp_canonical_url');
        }

        $query = $connection->query($this->_select);
        while ($row = $query->fetch()) {
            if ($useCanonicalUrlForCategory) {
                if ($row['wp_enable_canonical_url']) {
                    $canonicalUrl = $this->_parseCanonicalUrl($row['wp_canonical_url']);
                    if ($canonicalUrl) {
                        $row['url'] = $canonicalUrl;
                    }
                }
            } else {
                if ($row['wp_use_canonical_url_in_sitemap']) {
                    $canonicalUrl = $this->_parseCanonicalUrl($row['wp_canonical_url']);
                    if ($canonicalUrl) {
                        $row['url'] = $canonicalUrl;
                    }
                }
            }
            $category = $this->_prepareCategory($row);
            $categories[$category->getId()] = $category;
        }

        return $categories;
    }

    /**
     * Join attribute by code
     *
     * @param int $storeId
     * @param string $attributeCode
     * @param string $column Add attribute value to given column
     *
     * @return void
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function _joinAttribute($storeId, $attributeCode, $column = null)
    {

        $meta = $this->metadataPool->getMetadata(\Magento\Catalog\Api\Data\CategoryInterface::class);
        $linkField = $meta->getLinkField();

        if (!$this->_select instanceof \Magento\Framework\DB\Select) {
            return false;
        }

        if (!isset($this->_attributesCache[$attributeCode])) {
            $attribute = $this->_categoryResource->getAttribute($attributeCode);

            $this->_attributesCache[$attributeCode] = [
                'entity_type_id' => $attribute->getEntityTypeId(),
                'attribute_id' => $attribute->getId(),
                'table' => $attribute->getBackend()->getTable(),
                'is_global' => $attribute->getIsGlobal(),
                'backend_type' => $attribute->getBackendType(),
            ];
        }
        $attribute = $this->_attributesCache[$attributeCode];

        $connection = $this->getConnection();
        $attrTableAlias = 't1_' . $attributeCode;
        $this->_select->joinLeft(
            [$attrTableAlias => $attribute['table']],
            "e.{$linkField} = {$attrTableAlias}.{$linkField}"
            . ' AND ' . $connection->quoteInto($attrTableAlias . '.store_id = ?', Store::DEFAULT_STORE_ID)
            . ' AND ' . $connection->quoteInto($attrTableAlias . '.attribute_id = ?', $attribute['attribute_id']),
            []
        );
        // Global scope attribute value
        $columnValue = 't1_' . $attributeCode . '.value';

        if (!$attribute['is_global']) {
            $attrTableAlias2 = 't2_' . $attributeCode;
            $this->_select->joinLeft(
                ['t2_' . $attributeCode => $attribute['table']],
                "{$attrTableAlias}.{$linkField} = {$attrTableAlias2}.{$linkField}"
                . ' AND ' . $attrTableAlias . '.attribute_id = ' . $attrTableAlias2 . '.attribute_id'
                . ' AND ' . $connection->quoteInto($attrTableAlias2 . '.store_id = ?', $storeId),
                []
            );
            // Store scope attribute value
            $columnValue = $this->getConnection()->getIfNullSql('t2_'  . $attributeCode . '.value', $columnValue);
        }

        // Add attribute value to result set if needed
        if (isset($column)) {
            $this->_select->columns(
                [
                    $column => $columnValue
                ]
            );
        }
    }

    /**
     * @return \Magento\Framework\App\Config\ScopeConfigInterface
     */
    private function getScopeConfig() {
        if (!$this->scopeConfig) {
            $this->scopeConfig = ObjectManager::getInstance()
                ->get(\Magento\Framework\App\Config\ScopeConfigInterface::class);
        }
        return $this->scopeConfig;
    }

    /**
     * @param $storeId
     * @return bool
     */
    private function isCanonicalUrlForCategory($storeId)
    {
        $scopeConfig = $this->getScopeConfig();
        return $scopeConfig->getValue(
            'weltpixel_sitemap/general/use_canonical_url_for_category',
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param string $canonicalUrl
     * @return string
     */
    private function _parseCanonicalUrl($canonicalUrl) {
        $canonicalUrl = $canonicalUrl ?? '';
        $urlParts = parse_url($canonicalUrl);
        $canonicalUrl = isset($urlParts['path']) ? $urlParts['path'] : '';
        $canonicalUrl .= isset($urlParts['query']) ? "?".$urlParts['query'] : '';

        return $canonicalUrl;
    }

}
