<?php

namespace WeltPixel\SpeedOptimization\Observer;

use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use WeltPixel\SpeedOptimization\Helper\Data as SpeedHelper;
use WeltPixel\SpeedOptimization\Model\Storage;
use Magento\Framework\Filesystem;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\UrlInterface;

class CoreLayoutRenderElementObserver implements ObserverInterface
{
    /**
     * @var Storage
     */
    protected $storage;

    /**
     * @var SpeedHelper
     */
    protected $speedHelper;

    /**
     * @var Filesystem
     */
    protected $filesystem;

    /**
     * @var UrlInterface
     */
    protected $urlInterface;

    /**
     * CoreLayoutRenderElementObserver constructor.
     * @param SpeedHelper $speedHelper
     * @param Storage $storage
     * @param Filesystem $filesystem
     * @param UrlInterface $urlInterface
     */
    public function __construct(
        SpeedHelper $speedHelper,
        Storage $storage,
        Filesystem $filesystem,
        UrlInterface $urlInterface
    ) {
        $this->speedHelper = $speedHelper;
        $this->storage = $storage;
        $this->filesystem = $filesystem;
        $this->urlInterface = $urlInterface;
    }

    /**
     * Convert URL to filesystem path
     *
     * @param string $url
     * @return string|null
     */
    private function getFilePathFromUrl($url)
    {
        $baseUrl = $this->urlInterface->getBaseUrl(['_type' => UrlInterface::URL_TYPE_STATIC]);
        $filePath = str_replace($baseUrl, '', $url);
        return $filePath;
    }

    /**
     * @param Observer $observer
     * @return self
     */
    public function execute(Observer $observer)
    {
        $speedOptimizationsEabled = $this->speedHelper->isEnabled();
        $jsMoveToBottomEnabled = $this->speedHelper->isJsMoveToBottomEnabled();

        if (!$speedOptimizationsEabled) {
            return $this;
        }

        if ($jsMoveToBottomEnabled) {
            $elementName = $observer->getData('element_name');

            if ($elementName != 'wp.speedoptimization.before.body.end') {
                return $this;
            }

            $transport = $observer->getData('transport');
            $html = $transport->getOutput();

            $jsFiles = [];
            if ($this->storage->getJsFiles()) {
                $jsFiles = $this->storage->getJsFiles();
            }
            $jsScripts = '';

            $staticDirectory = $this->filesystem->getDirectoryRead(DirectoryList::STATIC_VIEW);

            foreach ($jsFiles as $attributes) {
                if (isset($attributes['integrity'])) {
                    $filePath = $this->getFilePathFromUrl($attributes['src']);

                    if ($filePath && $staticDirectory->isFile($filePath)) {
                        $fileContent = $staticDirectory->readFile($filePath);
                        $newHash = base64_encode(hash('sha256', $fileContent, true));
                        $attributes['integrity'] = "sha256-{$newHash}";
                    }
                }
                $jsScripts .= '<script';

                foreach ($attributes as $name => $value) {
                    $jsScripts .= ' ' . htmlspecialchars($name) . '="' . htmlspecialchars($value) . '"';
                }

                $jsScripts .= '></script>' . PHP_EOL;
            }
            $inlineScripts = $this->storage->getInlineScripts();

            $html = str_replace(['##js_scripts##', '##inline_scripts##'], [$jsScripts, $inlineScripts], $html);

            $transport->setOutput($html);
        }

        return $this;
    }
}
