<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingRules\Block\Zone;

use Magento\Checkout\Model\Session;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Directory\Model\ResourceModel\Country\CollectionFactory as CountryCollectionFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Phrase;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\View\Element\Template;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Address;
use MageWorx\GeoIP\Model\Geoip;
use MageWorx\ShippingRules\Helper\Data as Helper;
use MageWorx\ShippingRules\Helper\Image as ImageHelper;
use MageWorx\ShippingRules\Model\ExtendedZone;
use MageWorx\ShippingRules\Model\ResourceModel\ExtendedZone\Collection;
use MageWorx\ShippingRules\Model\ResourceModel\ExtendedZone\CollectionFactory as ExtendedZoneCollectionFactory;

class Selector extends Template
{

    /**
     * @var Session|\Magento\Backend\Model\Session\Quote
     */
    protected $session;

    /**
     * @var Geoip
     */
    protected $geoIp;

    /**
     * @var ExtendedZoneCollectionFactory
     */
    protected $ezCollectionFactory;

    /**
     * @var Helper
     */
    protected $helper;

    /**
     * @var CustomerSession
     */
    protected $customerSession;

    /**
     * @var Json
     */
    protected $jsonSerializer;

    /**
     * @var CountryCollectionFactory
     */
    protected $countryCollectionFactory;

    /**
     * @var ImageHelper
     */
    protected $imageHelper;

    /**
     * @param Template\Context $context
     * @param Session $checkoutSession
     * @param CustomerSession $customerSession
     * @param Geoip $geoip
     * @param ExtendedZoneCollectionFactory $ezCollectionFactory
     * @param Helper $helper
     * @param ImageHelper $imageHelper
     * @param Json $jsonSerializer
     * @param CountryCollectionFactory $countryCollectionFactory
     * @param array $data
     */
    public function __construct(
        Template\Context              $context,
        Session                       $checkoutSession,
        CustomerSession               $customerSession,
        Geoip                         $geoip,
        ExtendedZoneCollectionFactory $ezCollectionFactory,
        Helper                        $helper,
        ImageHelper                   $imageHelper,
        Json                          $jsonSerializer,
        CountryCollectionFactory      $countryCollectionFactory,
        array                         $data = []
    ) {
        parent::__construct($context, $data);

        $this->session         = $checkoutSession;
        $this->customerSession = $customerSession;
        $this->geoIp           = $geoip;
        $this->helper          = $helper;
        $this->imageHelper     = $imageHelper;
        $this->jsonSerializer  = $jsonSerializer;

        $this->ezCollectionFactory      = $ezCollectionFactory;
        $this->countryCollectionFactory = $countryCollectionFactory;
    }

    /**
     * JSON Data to create modal component
     *
     * @return string
     */
    public function getDataJson()
    {
        return $this->jsonSerializer->serialize(
            [
                'html'         => $this->getContent(),
                'country'      => $this->getShippingAddress()->getCountryModel()->getName(),
                'country_code' => $this->getShippingAddress()->getCountryId(),
                'country_list' => $this->getStoreCountryListAsOptionArray(),
                'save_url'     => $this->getUrl('mageworx_shippingrules/zone/change')
            ]
        );
    }

    /**
     * @return string
     */
    public function getContent()
    {
        $content = '<div id="shipping-zone-modal-public-content"></div>';

        return $content;
    }

    /**
     * Returns current customers shipping address from the quote
     *
     * @return Address
     */
    public function getShippingAddress()
    {
        /** @var Quote $quote */
        $quote = $this->session->getQuote();

        /** @var Address $shippingAddress */
        $shippingAddress = $quote->getShippingAddress();
        if (!$shippingAddress->getCountryId()) {
            $storedData = $this->customerSession->getData('customer_location');
            if (!empty($storedData) && is_array($storedData)) {
                $shippingAddress->addData($storedData);
            }
        }
        $shippingAddress->setCountryId((string)$shippingAddress->getCountryId());
        return $shippingAddress;
    }

    /**
     * Return store country list as an option array
     *
     * @return array
     */
    private function getStoreCountryListAsOptionArray()
    {
        $countryCollection = $this->countryCollectionFactory->create();
        $countryList       = $countryCollection->loadByStore()
                                               ->toOptionArray();

        return $countryList;
    }

    /**
     * Retrieve serialized JS layout configuration ready to use in template
     *
     * @return string
     * @throws NoSuchEntityException
     */
    public function getJsLayout()
    {
        $additionalData = [
            'components' => [
                'location' => [
                    'data' => [
                        'loc_test'             => 1,
                        'html'                 => $this->getContent(),
                        'country'              => $this->getShippingAddress()->getCountryModel()->getName(),
                        'country_code'         => $this->getShippingAddress()->getCountryId(),
                        'country_list'         => $this->getStoreCountryListAsOptionArray(),
                        'region'               => $this->getShippingAddress()->getRegion(),
                        'region_code'          => $this->getShippingAddress()->getRegionCode(),
                        'region_id'            => $this->getShippingAddress()->getRegionId(),
                        'save_url'             => $this->getUrl('mageworx_shippingrules/zone/change'),
                        'extended_zones'       => $this->getExtendedZones(),
                        'display_address_only' => $this->helper->isOnlyAddressFieldsShouldBeShown()
                    ],
                ],
            ],
        ];

        $this->jsLayout = array_merge_recursive($this->jsLayout, $additionalData);

        return $this->jsonSerializer->serialize($this->jsLayout);
    }

    /**
     * @return array|\MageWorx\ShippingRules\Model\ResourceModel\ExtendedZone[]
     * @throws NoSuchEntityException
     */
    private function getExtendedZones()
    {
        $outputItems = [];
        /** @var Collection $collection */
        $collection = $this->ezCollectionFactory->create();
        $collection->addIsActiveFilter();
        $collection->addStoreFilter($this->_storeManager->getStore()->getId());
        $collection->setOrder('priority');
        $items = $collection->getItems();
        /** @var ExtendedZone $item */
        foreach ($items as $item) {
            $outputItems[] = [
                'id'          => $item->getId(),
                'image'       => $this->imageHelper->getImageUrl($item->getImage(), ImageHelper::IMAGE_TYPE_FRONTEND_PREVIEW),
                'name'        => $item->getLabel($this->_storeManager->getStore()->getId()),
                'description' => $item->getDescription(),
                'countries'   => $item->getCountriesId(),
            ];
        }

        return $outputItems;
    }

    /**
     * @return Phrase
     */
    public function getLabel()
    {
        $currentCountry = $this->getCurrentCountry();
        if (!$currentCountry) {
            $customerData = $this->geoIp->getCurrentLocation();
            if ($customerData->getCode()) {
                $currentCountry = $this->getShippingAddress()
                                       ->getCountryModel()
                                       ->loadByCode($customerData->getCode())
                                       ->getName();
            }
        }

        if (!$currentCountry) {
            $label = __('Please, select you shipping region.');
        } else {
            $label = __('Your Shipping Country: %1', $currentCountry);
        }

        return $label;
    }

    /**
     * Get current country name
     *
     * @return string
     */
    public function getCurrentCountry()
    {
        return $this->getShippingAddress()->getCountryModel()->getName();
    }

    /**
     * Render block HTML
     *
     * @return string
     */
    protected function _toHtml()
    {
        if (!$this->helper->isEnabledPopup()) {
            return '';
        }

        return parent::_toHtml();
    }

    /**
     * Remove CSS files in case popup is disabled
     *
     * @return $this
     */
    protected function _prepareLayout()
    {
        $assetCollection = $this->pageConfig->getAssetCollection();
        if (!$this->helper->isEnabledPopup()) {
            $assetCollection->remove('MageWorx_ShippingRules::css/location.css');
            $assetCollection->remove('MageWorx_ShippingRules::css/slick.css');
            $assetCollection->remove('MageWorx_ShippingRules::css/slick-theme.css');
        }
        return parent::_prepareLayout();
    }
}
