<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingCalculatorBase\Block\Product\View\Calculator;

use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Model\Product as ProductModel;
use Magento\Checkout\Block\Checkout\LayoutProcessorInterface;
use Magento\Checkout\Model\CompositeConfigProvider;
use Magento\Directory\Model\Country\Postcode\ConfigInterface as PostcodeConfigInterface;
use Magento\Framework\Data\Form\FormKey;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Registry;
use Magento\Framework\Serialize\Serializer\JsonHexTag;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Framework\View\Element\Template;
use Magento\Framework\View\Element\Template\Context;
use MageWorx\ShippingCalculatorBase\Helper\Data as Helper;

/**
 * Class Container
 */
class Container extends Template
{
    /**
     * @var FormKey
     */
    protected $formKey;

    /**
     * @var array
     */
    protected $jsLayout;

    /**
     * @var CompositeConfigProvider
     */
    protected $configProvider;

    /**
     * @var array|LayoutProcessorInterface[]
     */
    protected $layoutProcessors;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var Registry
     */
    private $registry;

    /**
     * @var PostcodeConfigInterface
     */
    private $postCodesConfig;

    /**
     * @var Helper
     */
    private $helper;

    /**
     * Container constructor.
     *
     * @param Context $context
     * @param FormKey $formKey
     * @param CompositeConfigProvider $configProvider
     * @param PostcodeConfigInterface $postCodesConfig
     * @param JsonHexTag $serializer
     * @param Registry $registry
     * @param Helper $helper
     * @param array $layoutProcessors
     * @param array $data
     */
    public function __construct(
        Context $context,
        FormKey $formKey,
        CompositeConfigProvider $configProvider,
        PostcodeConfigInterface $postCodesConfig,
        JsonHexTag $serializer,
        Registry $registry,
        Helper $helper,
        array $layoutProcessors = [],
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->formKey          = $formKey;
        $this->jsLayout         = isset($data['jsLayout']) && is_array($data['jsLayout']) ? $data['jsLayout'] : [];
        $this->configProvider   = $configProvider;
        $this->postCodesConfig  = $postCodesConfig;
        $this->serializer       = $serializer;
        $this->registry         = $registry;
        $this->helper           = $helper;
        $this->layoutProcessors = $layoutProcessors;
    }

    /**
     * @inheritdoc
     */
    public function getJsLayout(): string
    {
        foreach ($this->layoutProcessors as $processor) {
            $this->jsLayout = $processor->process($this->jsLayout);
        }

        $layout = $this->serializer->serialize($this->jsLayout);
        if (!is_string($layout)) {
            $layout = '';
        }

        return $layout;
    }

    /**
     * Retrieve form key
     *
     * @return string
     */
    public function getFormKey(): string
    {
        return $this->formKey->getFormKey();
    }

    /**
     * Get base url for block.
     *
     * @return string
     */
    public function getBaseUrl(): string
    {
        return $this->_storeManager->getStore()->getBaseUrl();
    }

    /**
     * Retrieve serialized checkout config.
     *
     * @return string
     */
    public function getSerializedCheckoutConfig(): string
    {
        $config = $this->serializer->serialize($this->getCheckoutConfig());
        if (!is_string($config)) {
            $config = '';
        }

        return $config;
    }

    /**
     * Retrieve checkout configuration
     *
     * @return array
     */
    public function getCheckoutConfig(): array
    {
        try {
            $config = $this->configProvider->getConfig();
        } catch (NoSuchEntityException $exception) {
            $config = [];
        }

        $config['postCodes'] = $this->postCodesConfig->getPostCodes();

        return $config;
    }

    /**
     * Is block visible (available) on the product page
     * Must be invisible for the virtual, downloadable, out of stock products
     *
     * @return bool
     */
    public function isAvailable(): bool
    {
        if (!$this->helper->isEnabled()) {
            return false;
        }

        /** @var ProductModel $product */
        $product = $this->getCurrentProduct();
        if (!$product) {
            return false;
        }

        if (!$product->isAvailable() || !$product->isSalable()) {
            return false;
        }

        if ($product->isVirtual()) {
            return false;
        }

        if ($this->isRestrictedProduct($product)) {
            return false;
        }

        return true;
    }

    /**
     * @return ProductInterface|null
     */
    public function getCurrentProduct(): ?ProductInterface
    {
        $product = $this->registry->registry('current_product');

        return $product;
    }

    /**
     * @param ProductInterface|ProductModel $product
     * @return bool
     */
    public function isRestrictedProduct(ProductInterface $product): bool
    {
        if ($product->getData('hide_shipping_calculator')) {
            return true;
        }

        $category = $product->getCategory();
        if ($category && $category->getData('hide_shipping_calculator')) {
            return true;
        }

        return false;
    }

    /**
     * Is need to highlight the cheapest shipping method
     *
     * @return bool
     */
    public function isCheapestMethodHighlightNeeded(): bool
    {
        return $this->helper->isCheapestMethodHighlightNeeded();
    }
}
