/**
 * Copyright © MageWorx, Inc. All rights reserved.
 * See LICENSE.txt for license details.
 */
define([
    'underscore',
    'uiRegistry',
    'Magento_Ui/js/form/element/select',
    'Magento_Checkout/js/model/default-post-code-resolver'
], function (_, registry, Select, defaultPostCodeResolver) {
    'use strict';

    return Select.extend({
        defaults: {
            skipValidation: false,
            imports: {
                update: '${ $.parentName }.country_id:value'
            }
        },

        /**
         * Properties which will be set as an observable during initObservable method call
         */
        observableProperties: [
            'additionalClasses',
            'requiredEntry'
        ],

        initialize: function () {
            this._super();

            registry.async(this.parentName + '.' + 'country_id')(function (country) {
                if (country && !country.visible()) {
                    registry.async(this.customName)(function (customInput) {
                        if (customInput) {
                            customInput["validation"]["required-entry"] = true;
                        }
                    });
                }
            }.bind(this));

            return this;
        },

        /**
         * Initialize observables properties
         *
         * @returns {exports}
         */
        initObservable: function () {
            this._super();
            this.observe(this.observableProperties);
            this.visible.subscribe(function (value) {
                if (value && this.disabled()) {
                    this.visible(false);
                }
            }.bind(this));

            return this;
        },

        /**
         * @param {String} value
         */
        update: function (value) {
            var country = registry.get(this.parentName + '.' + 'country_id'),
                options = country.indexedOptions,
                isRegionRequired,
                option,
                skip = false,
                self = this;

            if (!value) {
                return;
            }
            option = options[value];

            if (typeof option === 'undefined') {
                return;
            }

            defaultPostCodeResolver.setUseDefaultPostCode(!option['is_zipcode_optional']);

            if (!this.requiredEntry() && !this.value() && country && country.visible()) {
                skip = true;
            }

            if (this.skipValidation || skip) {
                this.validation['required-entry'] = false;
                this.required(false);
                registry.get(this.customName, function (input) {
                    input.value('');
                });
            } else {
                if (option && !option['is_region_required'] && !this.requiredEntry()) {
                    this.error(false);
                    this.validation = _.omit(this.validation, 'required-entry');
                    registry.get(this.customName, function (input) {
                        input.validation['required-entry'] = false;
                        input.required(false);
                    });
                } else {
                    this.validation['required-entry'] = true;
                }

                if (option && !this.options().length) {
                    registry.get(this.customName, function (input) {
                        isRegionRequired = !!option['is_region_required'] || self.requiredEntry();
                        input.validation['required-entry'] = isRegionRequired;
                        input.validation['validate-not-number-first'] = true;
                        input.required(isRegionRequired);
                        input.value('');
                    });
                }

                this.required(!!option['is_region_required']);
            }
        },

        /**
         * Filters 'initialOptions' property by 'field' and 'value' passed,
         * calls 'setOptions' passing the result to it
         *
         * @param {*} value
         * @param {String} field
         */
        filter: function (value, field) {
            var superFn = this._super;

            registry.get(this.parentName + '.' + 'country_id', function (country) {
                var option = country.indexedOptions[value];

                superFn.call(this, value, field);

                if (option && option['is_region_visible'] === false) {
                    // hide select and corresponding text input field if region must not be shown for selected country
                    this.setVisible(false);

                    if (this.customEntry) {// eslint-disable-line max-depth
                        this.toggleInput(true);
                    }
                }
            }.bind(this));
        },

        /**
         * Change visibility for input.
         *
         * @param {Boolean} isVisible
         */
        toggleInput: function (isVisible) {
            this._super(isVisible && !this.disabled());
        },

        /**
         * Validates itself by it's validation rules using validator object.
         * If validation of a rule did not pass, writes it's message to
         * 'error' observable property.
         *
         * @returns {Object} Validate information.
         */
        validate: function () {
            var country = registry.get(this.parentName + '.' + 'country_id');
            if (!this.requiredEntry() && !this.value() && country && country.visible()) {
                return true;
            }

            return this._super();
        }
    });
});
