<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

declare(strict_types=1);

namespace MageWorx\ShippingCalculatorBase\Model;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\DataObjectFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Quote\Api\Data\CartInterfaceFactory;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Address\RateRequest;
use Magento\Quote\Model\Quote\Address\RateRequestFactory;
use Magento\Quote\Model\Quote\Item as QuoteItem;
use Magento\Shipping\Model\Shipping;
use Magento\Store\Model\StoreManagerInterface;
use MageWorx\ShippingCalculatorBase\Api\EstimateShippingMethodsInterface;
use MageWorx\ShippingCalculatorBase\Helper\Data as Helper;
use MageWorx\ShippingCalculatorBase\Plugin\ShippingRules\QuoteSubstitution as QuoteSubstitutionPlugin;
use Magento\Framework\Filter\Factory as FilterFactory;

/**
 * Class EstimateShippingMethods
 */
class EstimateShippingMethods implements EstimateShippingMethodsInterface
{
    /**
     * @var RateRequestFactory
     */
    private $rateRequestFactory;

    /**
     * @var Shipping
     */
    private $shipping;

    /**
     * @var CartInterfaceFactory
     */
    private $cartFactory;

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var DataObjectFactory
     */
    private $dataObjectFactory;

    /**
     * @var CartInterface
     */
    private $cart;

    /**
     * @var Helper
     */
    private $helper;

    /**
     * @var QuoteSubstitutionPlugin
     */
    private $quoteSubstitutionPlugin;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var FilterFactory
     */
    private $filterFactory;

    /**
     * EstimateShippingMethods constructor.
     *
     * @param RateRequestFactory $rateRequestFactory
     * @param Shipping $shipping
     * @param CartInterfaceFactory $cartFactory
     * @param ProductRepositoryInterface $productRepository
     * @param DataObjectFactory $dataObjectFactory
     * @param Helper $helper
     * @param QuoteSubstitutionPlugin $quoteSubstitutionPlugin
     * @param StoreManagerInterface $storeManager
     * @param FilterFactory $filterFactory
     */
    public function __construct(
        RateRequestFactory $rateRequestFactory,
        Shipping $shipping,
        CartInterfaceFactory $cartFactory,
        ProductRepositoryInterface $productRepository,
        DataObjectFactory $dataObjectFactory,
        Helper $helper,
        QuoteSubstitutionPlugin $quoteSubstitutionPlugin,
        StoreManagerInterface $storeManager,
        FilterFactory $filterFactory
    ) {
        $this->rateRequestFactory      = $rateRequestFactory;
        $this->shipping                = $shipping;
        $this->cartFactory             = $cartFactory;
        $this->productRepository       = $productRepository;
        $this->dataObjectFactory       = $dataObjectFactory;
        $this->helper                  = $helper;
        $this->quoteSubstitutionPlugin = $quoteSubstitutionPlugin;
        $this->storeManager            = $storeManager;
        $this->filterFactory           = $filterFactory;
    }

    /**
     * Estimate shipping methods
     *
     * @param array $addressData
     * @param array $itemData
     * @return array
     * @throws LocalizedException
     */
    public function estimate(array $addressData, array $itemData): array
    {
        /** @var \Magento\Quote\Model\Quote $cart */
        $cart    = $this->getCart();
        $request = $this->getRatesRequest($addressData);
        /** @var \Magento\Quote\Model\Quote\Address $shippingAddress */
        $shippingAddress = $cart->getShippingAddress();
        $shippingAddress->addData($addressData);

        $this->addItemToRequest($itemData, $request);
        $this->calculatePackageQty($request);
        $this->quoteSubstitutionPlugin->setAddress($shippingAddress);

        $cart->collectTotals();

        $this->updateRequestValues($request, $shippingAddress);
        $this->shipping->collectRates($request);
        $result = $this->shipping->getResult();
        $rates  = $result->sortRatesByPrice();

        $preparedRates = $this->prepareRatesResponse($rates);

        return $preparedRates;
    }

    /**
     * Set totals and weight to request
     *
     * @param $request
     * @param $address
     */
    private function updateRequestValues($request, $address): void
    {
        $request->setPackageValue($address->getBaseSubtotal());
        $request->setPackagePhysicalValue($address->getBaseSubtotal());
        $request->setPackageValueWithDiscount($address->getBaseSubtotalWithDiscount());
        $request->setPackageWeight($address->getWeight());
        $request->setFreeMethodWeight($address->getFreeMethodWeight());
        $request->setBaseSubtotalInclTax($address->getBaseSubtotalInclTax());
    }

    /**
     * Create empty cart instance
     *
     * @return CartInterface
     */
    private function createCart(): CartInterface
    {
        /** @var CartInterface|Quote $cart */
        $cart = $this->cartFactory->create();

        /**
         * Prevent unnecessary saving.
         *
         * @see \MageWorx\ShippingCalculatorBase\Plugin\Quote::afterIsSaveAllowed()
         */
        $cart->setData(static::QUOTE_SAVE_DISABLED_FLAG, true);

        return $cart;
    }

    /**
     * Get current cart instance or create new
     *
     * @return CartInterface
     */
    private function getCart(): CartInterface
    {
        if (!$this->cart) {
            $this->cart = $this->createCart();
        }

        return $this->cart;
    }

    /**
     * @param array $addressData
     * @return RateRequest
     */
    private function getRatesRequest(array $addressData): RateRequest
    {
        /** @var RateRequest $request */
        $request = $this->rateRequestFactory->create(['data' => $addressData]);

        /** @var \Magento\Store\Api\Data\StoreInterface $store */
        $store = $this->storeManager->getStore();
        $request->setStoreId($store->getId());
        $request->setWebsiteId($store->getWebsiteId());
        $request->setBaseCurrency($store->getBaseCurrency());
        $request->setPackageCurrency($store->getCurrentCurrency());
        $request->setLimitCarrier('');

        return $request;
    }

    /**
     * Add item to the rate request using input params
     *
     * @param array $itemData
     * @param RateRequest $request
     * @throws LocalizedException
     */
    private function addItemToRequest(array $itemData, RateRequest $request): void
    {
        $productId = $itemData['product'] ?? null;
        if (!$productId) {
            throw new LocalizedException(__('Product id is missed.'));
        }

        /** @var \Magento\Catalog\Model\Product $product */
        $product = $this->productRepository->getById($productId);

        $itemDataObject = $this->dataObjectFactory->create(['data' => $itemData]);
        $cart           = $this->getCart();
        $cart->addProduct($product, $itemDataObject);
        $items = $cart->getAllVisibleItems();
        $request->setAllItems($items);
    }

    /**
     * @param RateRequest $request
     * @return float
     */
    private function calculatePackageQty(RateRequest $request): float
    {
        /** @var QuoteItem[] $items */
        $items = $request->getAllItems() ?? [];
        $qty   = 0;
        /** @var QuoteItem $item */
        foreach ($items as $item) {
            $qty += $item->getQty();
        }

        $request->setPackageQty($qty);

        return $qty;
    }

    /**
     * @param array $rateResult
     * @return array
     */
    private function prepareRatesResponse(\Magento\Shipping\Model\Rate\Result $rateResult): array
    {
        $response                     = [];
        $shippingMethodsConfiguration = $this->helper->getShippingMethodsConfiguration();

        $rates = $rateResult->getAllRates();

        if ($this->storeManager->getStore()->getBaseCurrency()
            && $this->storeManager->getStore()->getCurrentCurrency()
        ) {
            $currencyFilter = $this->storeManager->getStore()->getCurrentCurrency()->getFilter();
            $currencyFilter->setRate(
                $this->storeManager->getStore()->getBaseCurrency()->getRate(
                    $this->storeManager->getStore()->getCurrentCurrency()
                )
            );
        } elseif ($this->storeManager->getStore()->getDefaultCurrency()) {
            $currencyFilter = $this->storeManager->getStore()->getDefaultCurrency()->getFilter();
        } else {
            $currencyFilter = $this->filterFactory->createFilter(
                'sprintf',
                [
                    'format' => '%s',
                    'decimals' => 2
                ]
            );
        }

        /** @var \Magento\Quote\Model\Quote\Address\RateResult\Method $rate */
        foreach ($rates as $rate) {
            $carrierCode = $rate->getData('carrier');
            $methodCode  = $rate->getData('method');
            $methodData  = $rate->getData();

            $methodDescription = $methodData['description'] ?? '';
            $deliveryFormatted = '';
            $imageLink         = '';
            $code              = $carrierCode . '_' . $methodCode;
            $price             = $methodData['price'] ?? '';
            $carrierTitle      = $methodData['carrier_title'] ?? '';

            if (isset($methodData['method_title'])) {
                $title = $methodData['method_title'];
            } else {
                $title = '';
            }

            if (isset($methodData['price'])) {
                $priceFormatted = $currencyFilter->filter($methodData['price']);
            } else {
                $priceFormatted = '';
            }

            if ($carrierCode == 'freeshipping') {
                $code           = 'freeshipping_freeshipping';
                $title          = $title ?? __('Free Shipping');
                $priceFormatted = __('Free');
            }

            $data = [
                'code'               => $code,
                'name'               => $code,
                'title'              => $title,
                'carrier_title'      => $carrierTitle,
                'price_formatted'    => $priceFormatted,
                'price'              => $price,
                'description'        => $methodDescription,
                'image'              => $imageLink,
                'delivery_formatted' => $deliveryFormatted
            ];

            if (!empty($shippingMethodsConfiguration[$carrierCode])) {
                $data = array_merge($data, $shippingMethodsConfiguration[$carrierCode]);
            }

            if (!empty($shippingMethodsConfiguration[$code])) {
                $data = array_merge($data, $shippingMethodsConfiguration[$code]);
            }

            if (!empty($methodData['error_message'])) {
                $data['error_message'] = $methodData['error_message'];
            } else {
                $data['error_message'] = '';
            }

            $response[] = $data;
        }

        return $response;
    }
}
