<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\OptionBase\Plugin;

use Closure;
use Magento\Catalog\Model\Product\Option\Value as OptionValue;
use Magento\Catalog\Model\ResourceModel\Product\Option\Collection as OptionCollection;
use Magento\Catalog\Model\ResourceModel\Product\Option\Value\Collection as OptionValueCollection;
use Magento\Catalog\Model\ResourceModel\Product\Option\Value\CollectionFactory as OptionValueCollectionFactory;
use Magento\Customer\Model\Group;
use Magento\Framework\App\State;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Serialize\Serializer\Json as Serializer;
use Magento\Store\Model\StoreManagerInterface as StoreManager;
use MageWorx\OptionBase\Helper\Data as BaseHelper;
use MageWorx\OptionBase\Helper\System as SystemHelper;
use MageWorx\OptionBase\Model\ConditionValidator;
use MageWorx\OptionBase\Model\ResourceModel\CollectionUpdaterRegistry;

class CollectProductOptionConditions
{
    protected State $state;
    private CollectionUpdaterRegistry $collectionUpdaterRegistry;
    protected OptionValueCollectionFactory $optionValueCollectionFactory;
    protected StoreManager $storeManager;
    protected SystemHelper $systemHelper;
    protected array $valuesCollectionCache = [];
    protected Serializer $serializer;
    protected ConditionValidator $conditionValidator;
    protected int $customerGroupId;
    protected ManagerInterface $eventManager;
    protected BaseHelper $baseHelper;

    public function __construct(
        CollectionUpdaterRegistry $collectionUpdaterRegistry,
        OptionValueCollectionFactory $optionValueCollectionFactory,
        SystemHelper $systemHelper,
        StoreManager $storeManager,
        State $state,
        Serializer $serializer,
        ConditionValidator $conditionValidator,
        ManagerInterface $eventManager,
        BaseHelper $baseHelper
    ) {
        $this->collectionUpdaterRegistry    = $collectionUpdaterRegistry;
        $this->optionValueCollectionFactory = $optionValueCollectionFactory;
        $this->systemHelper                 = $systemHelper;
        $this->storeManager                 = $storeManager;
        $this->state                        = $state;
        $this->serializer                   = $serializer;
        $this->conditionValidator           = $conditionValidator;
        $this->eventManager                 = $eventManager;
        $this->baseHelper                   = $baseHelper;
    }

    /**
     * Set product ID to collection updater registry for future use in collection updaters
     */
    public function beforeGetProductOptions(
        OptionCollection $object,
        ?int             $productId,
        ?int             $storeId,
        ?bool            $requiredOnly = false
    ): array {
        $this->collectionUpdaterRegistry->setCurrentEntityIds([$productId]);
        $this->collectionUpdaterRegistry->setCurrentEntityType('product');

        if ($this->systemHelper->isOptionImportAction()) {
            $this->collectionUpdaterRegistry->setOptionIds([]);
            $this->collectionUpdaterRegistry->setOptionValueIds([]);
        }

        return [$productId, $storeId, $requiredOnly];
    }

    /**
     * Set option/option value IDs to collection updater registry for future use in collection updaters
     */
    public function aroundAddValuesToResult(
        OptionCollection $subject,
        Closure $proceed,
        ?int $storeId = null
    ): OptionCollection {
        \Magento\Framework\Profiler::start('APO_CPO_aroundAddValuesToResult');

        if ($storeId === null) {
            $storeId = (int)$this->storeManager->getStore()->getId();
        }

        \Magento\Framework\Profiler::start('APO_CPO_getOptionIds');
        $optionIds = $this->getOptionIdsFromCollection($subject);
        \Magento\Framework\Profiler::stop('APO_CPO_getOptionIds');

        if ($optionIds) {
            $this->collectionUpdaterRegistry->setOptionIds($optionIds);

            \Magento\Framework\Profiler::start('APO_CPO_getOptionValues');
            $values = $this->getOptionValues($optionIds, $storeId);
            \Magento\Framework\Profiler::stop('APO_CPO_getOptionValues');

            $valueIds      = [];
            $groupedValues = [];

            $isGraphQlRequest = ($this->state->getAreaCode() === 'graphql');
            if ($isGraphQlRequest) {
                $this->customerGroupId = (int)$this->systemHelper->resolveCurrentCustomerGroupId();
            }

            \Magento\Framework\Profiler::start('APO_CPO_groupOptionValues');
            foreach ($values as $value) {
                if (!$value->getOptionTypeId()) {
                    continue;
                }
                $valueIds[]                             = $value->getOptionTypeId();
                $groupedValues[$value->getOptionId()][] = $value;
            }
            \Magento\Framework\Profiler::stop('APO_CPO_groupOptionValues');

            \Magento\Framework\Profiler::start('APO_CPO_attachValues');
            $this->attachValuesToOptions($subject, $groupedValues, $isGraphQlRequest);
            \Magento\Framework\Profiler::stop('APO_CPO_attachValues');

            if ($this->baseHelper->isModuleEnabled('Magento_InventorySalesAdminUi')) {
                \Magento\Framework\Profiler::start('APO_CPO_dispatchQtyUpdate');
                $this->eventManager->dispatch(
                    'mageworx_optioninventory_linked_qty_source_update',
                    ['data_to_update' => $values, 'option_collection' => $subject]
                );
                \Magento\Framework\Profiler::stop('APO_CPO_dispatchQtyUpdate');
            }

            if ($valueIds) {
                $this->collectionUpdaterRegistry->setOptionValueIds($valueIds);
            }
        }

        \Magento\Framework\Profiler::stop('APO_CPO_aroundAddValuesToResult');

        return $subject;
    }

    /**
     * Set Advanced Pricing price for current customer group
     */
    protected function setPriceForCurrentCustomerGroup(string $priceType, OptionValue $value): void
    {
        $optionValuePrice             = $value->getDataByKey('default_price');
        $pricesAdvancedPricing        = $value->getDataByKey($priceType);
        $decodedPricesAdvancedPricing = $pricesAdvancedPricing ? $this->serializer->unserialize($pricesAdvancedPricing) : null;

        if ($decodedPricesAdvancedPricing && is_int($this->customerGroupId)) {
            $priceForCurrentCustomerGroup = [];
            $priceForAllGroups            = [];

            foreach ($decodedPricesAdvancedPricing as $priceItem) {
                if (!$this->conditionValidator->isValidated($priceItem, $optionValuePrice)) {
                    continue;
                }

                switch ($priceItem['customer_group_id']) {
                    case $this->customerGroupId:
                        $priceForCurrentCustomerGroup = $priceItem;
                        break;
                    case Group::CUST_GROUP_ALL:
                        $priceForAllGroups = $priceItem;
                        break;
                }
            }

            $resultPrice = null;
            if (!empty($priceForCurrentCustomerGroup)) {
                $resultPrice = $this->serializer->serialize($priceForCurrentCustomerGroup);
            } elseif (!empty($priceForAllGroups)) {
                $resultPrice = $this->serializer->serialize($priceForAllGroups);
            }

            $value->setData($priceType, $resultPrice);
        }
    }

    /**
     * Get option IDs from option collection
     */
    protected function getOptionIdsFromCollection(OptionCollection $optionCollection): array
    {
        $optionIds = [];
        foreach ($optionCollection as $option) {
            if ($option->getId()) {
                $optionIds[] = $option->getId();
            }
        }
        return $optionIds;
    }

    /**
     * Load option values from database without SQL sorting
     */
    protected function getOptionValues(array $optionIds, int $storeId): OptionValueCollection
    {
        return $this->optionValueCollectionFactory->create()
                                                  ->addTitleToResult($storeId)
                                                  ->addPriceToResult($storeId)
                                                  ->addOptionToFilter($optionIds);
    }

    /**
     * Attach values to options and sort values in PHP
     */
    protected function attachValuesToOptions(OptionCollection $optionCollection, array $groupedValues, bool $isGraphQlRequest): void
    {
        foreach ($optionCollection as $option) {
            $optionId = $option->getId();
            if (!isset($groupedValues[$optionId])) {
                continue;
            }

            $values = $groupedValues[$optionId];

            usort(
                $values,
                static fn(OptionValue $a, OptionValue $b) => [$a->getSortOrder(), $a->getTitle()] <=> [$b->getSortOrder(), $b->getTitle()]
            );

            foreach ($values as $value) {
                if ($isGraphQlRequest) {
                    $this->setPriceForCurrentCustomerGroup('tier_price', $value);
                    $this->setPriceForCurrentCustomerGroup('special_price', $value);
                }
                $option->addValue($value);
                $value->setOption($option);
            }
        }
    }
}
