<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */
declare(strict_types = 1);

namespace MageWorx\ShippingRules\Test\Unit\CustomerAttributes\CustomerAttributesFilter;

use Magento\Eav\Model\Entity\Attribute;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\TestFramework\Unit\Helper\ObjectManager;
use MageWorx\ShippingRules\Model\Rule\Attributes\Customer\Filters\ByConfig;
use PHPUnit\Framework\TestCase;

class ByConfigTest extends TestCase
{
    public const TESTED_CLASS_NAME = 'MageWorx\ShippingRules\Model\Rule\Attributes\Customer\Filters\ByConfig';

    /**
     * @var ByConfig
     */
    private $model;

    private $attributeMock;
    private $scopeConfigMock;

    /**
     * @inheritdoc
     */
    public function setUp(): void
    {
        $objectManager = new ObjectManager($this);

        $this->attributeMock = $this->getMockBuilder(Attribute::class)
                                    ->disableOriginalConstructor()
                                    ->getMock();

        $this->scopeConfigMock = $this->getMockBuilder(ScopeConfigInterface::class)
                                      ->disableOriginalConstructor()
                                      ->getMock();

        $this->model = $objectManager->getObject(
            static::TESTED_CLASS_NAME,
            [
                'scopeConfig' => $this->scopeConfigMock
            ]
        );
    }

    /**
     * @return void
     */
    public function testReturnCorrectValueWhenAttributeInDisabledListAsString()
    {
        $this->scopeConfigMock->expects($this->once())
                              ->method('getValue')
                              ->with($this->model::XML_PATH_DISABLED_CUSTOMER_ATTRIBUTES)
                              ->willReturn('disabled_attribute_1,disabled_attribute_2,disabled_attribute_3');
        $this->attributeMock->expects($this->atLeastOnce())
                            ->method('getAttributeCode')
                            ->willReturn('disabled_attribute_2');

        $result = $this->model->check($this->attributeMock);

        $this->assertFalse($result);
    }

    /**
     * @return void
     */
    public function test2ReturnCorrectValueWhenAttributeInDisabledListAsString()
    {
        $this->scopeConfigMock->expects($this->once())
                              ->method('getValue')
                              ->with($this->model::XML_PATH_DISABLED_CUSTOMER_ATTRIBUTES)
                              ->willReturn('disabled_attribute_1,disabled_attribute_2,disabled_attribute_3');
        $this->attributeMock->expects($this->atLeastOnce())
                            ->method('getAttributeCode')
                            ->willReturn('disabled_attribute_100');

        $result = $this->model->check($this->attributeMock);

        $this->assertTrue($result);
    }

    /**
     * @return void
     */
    public function testReturnCorrectValueWhenAttributeInDisabledListAsArray()
    {
        $this->scopeConfigMock->expects($this->once())
                              ->method('getValue')
                              ->with($this->model::XML_PATH_DISABLED_CUSTOMER_ATTRIBUTES)
                              ->willReturn(['disabled_attribute_100', 'disabled_attribute_200']);
        $this->attributeMock->expects($this->atLeastOnce())
                            ->method('getAttributeCode')
                            ->willReturn('disabled_attribute_200');

        $result = $this->model->check($this->attributeMock);

        $this->assertFalse($result);
    }

    /**
     * @return void
     */
    public function test2ReturnCorrectValueWhenAttributeInDisabledListAsArray()
    {
        $this->scopeConfigMock->expects($this->once())
                              ->method('getValue')
                              ->with($this->model::XML_PATH_DISABLED_CUSTOMER_ATTRIBUTES)
                              ->willReturn(['disabled_attribute_100', 'disabled_attribute_200']);
        $this->attributeMock->expects($this->atLeastOnce())
                            ->method('getAttributeCode')
                            ->willReturn('disabled_attribute_9');

        $result = $this->model->check($this->attributeMock);

        $this->assertTrue($result);
    }

    /**
     * @return void
     */
    public function testConfigValues()
    {
        $this->scopeConfigMock->expects($this->once())
                              ->method('getValue')
                              ->with($this->model::XML_PATH_DISABLED_CUSTOMER_ATTRIBUTES)
                              ->willReturn(
                                  'disabled_attribute_1,disabled_attribute_2,disabled_attribute_3, disabled_attribute_4'
                              );

        $result = $this->model->getDisabledAttributes();

        $this->assertArrayHasKey(0, $result);
        $this->assertArrayHasKey(1, $result);
        $this->assertArrayHasKey(2, $result);
        $this->assertArrayHasKey(3, $result);
        $this->assertArrayNotHasKey(4, $result);
    }

    /**
     * @return void
     */
    public function testDisabledAttributesFromConfigType()
    {
        $result = $this->model->getDisabledAttributes();

        $this->assertIsArray($result);
        $this->assertEmpty($result);
    }
}
