<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingRules\Model\ResourceModel;

use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Framework\Stdlib\StringUtils;
use Magento\Rule\Model\ResourceModel\AbstractResource;
use MageWorx\ShippingRules\Model\Zone as ZoneModel;

class Zone extends AbstractResource
{
    /**
     * Store associated with zone entities information map
     *
     * @var array
     */
    protected $_associatedEntitiesMap = [
        'store' => [
            'associations_table' => ZoneModel::ZONE_STORE_TABLE_NAME,
            'rule_id_field'      => 'zone_id',
            'entity_id_field'    => 'store_id',
        ],
    ];

    /**
     * @var array
     */
    protected $storeIds = [];

    /**
     * Magento string lib
     *
     * @var StringUtils
     */
    protected $string;

    /**
     * @param Context $context
     * @param StringUtils $string
     * @param null $connectionName
     */
    public function __construct(
        Context     $context,
        StringUtils $string,
                    $connectionName = null
    ) {
        $this->string = $string;
        parent::__construct($context, $connectionName);
    }

    /**
     * Initialize main table and table id field
     *
     * @return void
     */
    protected function _construct()
    {
        $this->_init(ZoneModel::ZONE_TABLE_NAME, 'entity_id');
    }

    /**
     * Add store ids to zone data after load
     *
     * @param AbstractModel $object
     * @return $this
     */
    protected function _afterLoad(AbstractModel $object)
    {
        $this->loadStoreIds($object);

        return parent::_afterLoad($object);
    }

    /**
     * @param AbstractModel $object
     * @return void
     */
    public function loadStoreIds(AbstractModel $object)
    {
        $this->storeIds = (array)$this->getStoreIds($object->getId());
        $object->setData('store_ids', $this->storeIds);
    }

    /**
     * Retrieve store ids of specified zone
     *
     * @param int $zoneId
     * @return array
     */
    public function getStoreIds($zoneId)
    {
        return $this->getAssociatedEntityIds($zoneId, 'store');
    }

    /**
     * Bind shipping zone to the store(s).
     *
     * @param AbstractModel $object
     * @return AbstractDb
     */
    protected function _afterSave(AbstractModel $object)
    {
        if ($object->hasStoreIds()) {
            $storeIds = $object->getStoreIds();
            if (!is_array($storeIds)) {
                $storeIds = explode(',', (string)$storeIds);
            }
            $this->bindRuleToEntity($object->getId(), $storeIds, 'store');
        }

        return parent::_afterSave($object);
    }
}
