<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingRules\Model\ImportExport;

use Magento\Framework\Api\CustomAttributesDataInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\DataObject;
use Magento\Framework\DataObject\Factory;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\File\Csv;
use Magento\Framework\Message\ManagerInterface;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\Reflection\MethodsMap;
use Magento\Framework\UrlInterface;
use Magento\Store\Model\StoreManagerInterface;
use MageWorx\ShippingRules\Api\CarrierRepositoryInterface;
use MageWorx\ShippingRules\Api\Data\CarrierInterface;
use MageWorx\ShippingRules\Api\Data\MethodInterface;
use MageWorx\ShippingRules\Api\Data\RateInterface;
use MageWorx\ShippingRules\Api\ImportExportEntity;
use MageWorx\ShippingRules\Api\MethodRepositoryInterface;
use MageWorx\ShippingRules\Api\RateRepositoryInterface;
use MageWorx\ShippingRules\Helper\Data;
use ReflectionException;

/**
 * Class AbstractImportExport
 */
abstract class AbstractImportExport extends DataObject
{
    public const CARRIER_ENTITY_KEY = 'carrier';
    public const METHOD_ENTITY_KEY  = 'method';
    public const RATE_ENTITY_KEY    = 'rate';
    public const ENTITY_TYPE        = 'ie_type';
    public const ENTITY_ID          = 'id';

    public const COLON = '"';

    /**
     * @var MethodsMap
     */
    protected $reflectionMethodsMap;

    /**
     * @var array
     */
    protected $entitiesMap = [
        self::CARRIER_ENTITY_KEY => 'MageWorx\ShippingRules\Api\Data\CarrierInterface',
        self::METHOD_ENTITY_KEY  => 'MageWorx\ShippingRules\Api\Data\MethodInterface',
        self::RATE_ENTITY_KEY    => 'MageWorx\ShippingRules\Api\Data\RateInterface',
    ];

    /**
     * @var array
     */
    protected $entitiesDependency = [
        self::RATE_ENTITY_KEY   => self::METHOD_ENTITY_KEY,
        self::METHOD_ENTITY_KEY => self::CARRIER_ENTITY_KEY
    ];

    /**
     * @var CarrierRepositoryInterface
     */
    protected $carrierRepository;

    /**
     * @var MethodRepositoryInterface
     */
    protected $methodRepository;

    /**
     * @var RateRepositoryInterface
     */
    protected $rateRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    protected $searchCriteriaBuilder;

    /**
     * @var array|CarrierInterface[]
     */
    protected $carriers;

    /**
     * @var array|MethodInterface[]
     */
    protected $methods;

    /**
     * @var array|RateInterface[]
     */
    protected $rates;

    /**
     * Data fields cache
     *
     * @var array
     */
    protected $dataFields;

    /**
     * Data fields cache stored by entity
     *
     * @var array
     */
    protected $dataFieldsByEntity;

    /**
     * @var Factory
     */
    protected $dataObjectFactory;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var Csv
     */
    protected $csvProcessor;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var ObjectManagerInterface
     */
    protected $objectManager;

    /**
     * @var ManagerInterface
     */
    protected $messageManager;

    /**
     * @var EventManager
     */
    protected $eventManager;

    /**
     * AggregatedDataObject constructor.
     *
     * @param MethodsMap $reflectionMethodsMap
     * @param CarrierRepositoryInterface $carrierRepository
     * @param MethodRepositoryInterface $methodRepository
     * @param RateRepositoryInterface $rateRepository
     * @param Data $helper
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param Factory $dataObjectFactory
     * @param StoreManagerInterface $storeManager
     * @param Csv $csvProcessor
     * @param ObjectManagerInterface $objectManager
     * @param ManagerInterface $messageManager
     * @param EventManager $eventManager
     * @param array $entities
     * @param array $data
     */
    public function __construct(
        MethodsMap                 $reflectionMethodsMap,
        CarrierRepositoryInterface $carrierRepository,
        MethodRepositoryInterface  $methodRepository,
        RateRepositoryInterface    $rateRepository,
        Data                       $helper,
        SearchCriteriaBuilder      $searchCriteriaBuilder,
        Factory                    $dataObjectFactory,
        StoreManagerInterface      $storeManager,
        Csv                        $csvProcessor,
        ObjectManagerInterface     $objectManager,
        ManagerInterface           $messageManager,
        EventManager               $eventManager,
        array                      $entities = [],
        array                      $data = []
    ) {
        parent::__construct($data);
        $this->reflectionMethodsMap  = $reflectionMethodsMap;
        $this->carrierRepository     = $carrierRepository;
        $this->methodRepository      = $methodRepository;
        $this->rateRepository        = $rateRepository;
        $this->helper                = $helper;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->dataObjectFactory     = $dataObjectFactory;
        $this->storeManager          = $storeManager;
        $this->csvProcessor          = $csvProcessor;
        $this->objectManager         = $objectManager;
        $this->messageManager        = $messageManager;
        $this->eventManager          = $eventManager;
    }

    /**
     * Get headers for the selected entities
     *
     * @param array $entities
     * @return DataObject
     */
    protected function getHeaders($entities = [])
    {
        $dataFields = $this->getDataFields($entities);
        $dataObject = $this->dataObjectFactory->create($dataFields);

        return $dataObject;
    }

    /**
     * Get all available fields for the selected entities (all available by default)
     *
     * @param array $entities
     * @return array
     */
    protected function getDataFields(array $entities = [])
    {
        asort($entities);
        $cacheKey = implode(':', $entities);

        if (!empty($this->dataFields[$cacheKey])) {
            return $this->dataFields[$cacheKey];
        }

        if (empty($entities)) {
            $entities = $this->getAllAvailableEntitiesName();
        }

        $fields = [static::ENTITY_TYPE => __('IE Type')];
        if ($this->useIds()) {
            $fields[static::ENTITY_ID] = __('ID');
        }

        foreach ($entities as $entityName) {
            if (empty($this->entitiesMap[$entityName])) {
                continue;
            }

            $entityInterfaceName = $this->entitiesMap[$entityName];
            $methods             = $this->reflectionMethodsMap->getMethodsMap($entityInterfaceName);

            // Detect ignored columns
            $classInstance = $this->objectManager->get($entityInterfaceName);
            if ($classInstance instanceof ImportExportEntity) {
                $ignoredColumns = $classInstance::getIgnoredColumnsForImportExport();
            } else {
                $ignoredColumns = [];
            }

            if ($classInstance instanceof CustomAttributesDataInterface) {
                $extensionInterfaceName = preg_replace('/Interface$/ui', 'ExtensionInterface', $entityInterfaceName);
                try {
                    $extensionMethods = $this->reflectionMethodsMap->getMethodsMap($extensionInterfaceName);
                    $methods          += $extensionMethods;
                } catch (ReflectionException $reflectionException) {
                    // Do nothing, just show error message
                    $this->messageManager->addNoticeMessage(
                        __('Incorrect extension interface %1 :', $extensionInterfaceName)
                    );
                    $this->messageManager->addNoticeMessage($reflectionException->getMessage());
                }
            }

            foreach ($methods as $methodName => $data) {
                $propertyKeyName = $this->_underscore(substr($methodName, 3));
                if (in_array($propertyKeyName, $ignoredColumns)) {
                    continue;
                }
                $humanReadableKeyName     = ucwords(str_replace('_', ' ', $propertyKeyName));
                $fields[$propertyKeyName] = $humanReadableKeyName;
            }

            // Save data fields cache specified by each entity (separated)
            // @DEBUG
            $this->dataFieldsByEntity[$entityName] = $fields;
        }

        $this->dataFields[$cacheKey] = $fields;

        return $this->dataFields[$cacheKey];
    }

    /**
     * Get name of the all entities available by default
     *
     * @return array
     */
    protected function getAllAvailableEntitiesName()
    {
        return [
            self::CARRIER_ENTITY_KEY,
            self::METHOD_ENTITY_KEY,
            self::RATE_ENTITY_KEY
        ];
    }

    /**
     * Should or not use ids during import/export
     *
     * @return bool
     */
    protected function useIds()
    {
        return $this->helper->isIdsUsedDuringImport();
    }

    protected function _underscore($name)
    {
        if (isset(self::$_underscoreCache[$name])) {
            return self::$_underscoreCache[$name];
        }

        $result = strtolower(trim(preg_replace('/([A-Z]|[0-9]+)/', "_$1", $name), '_'));

        self::$_underscoreCache[$name] = $result;

        return $result;
    }

    /**
     * Get creation time
     *
     * @return int
     */
    protected function getTime()
    {
        return time();
    }

    /**
     * Get store base url
     *
     * @return string
     * @throws NoSuchEntityException
     */
    protected function getHost()
    {
        return $this->storeManager->getStore()->getBaseUrl(UrlInterface::URL_TYPE_WEB);
    }
}
