<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingCalculatorBase\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\UrlInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Framework\Math\Random;
use Magento\Framework\Serialize\Serializer\Json as SerializeJson;
use Psr\Log\LoggerInterface as Logger;
use Magento\Tax\Model\Config as TaxConfig;

/**
 * Class Data
 */
class Data extends AbstractHelper
{
    const XML_PATH_ENABLED                    = 'mageworx_shipping_calculator/main/enabled';
    const XML_PATH_CALCULATE_ON_LOAD          = 'mageworx_shipping_calculator/main/calculate_on_load';
    const XML_PATH_CALCULATE_BY_COUNTRY       = 'mageworx_shipping_calculator/main/calculate_by_country';
    const XML_PATH_COUNTRY_REQUIRED           = 'mageworx_shipping_calculator/main/country_required';
    const XML_PATH_CALCULATE_BY_REGION        = 'mageworx_shipping_calculator/main/calculate_by_region';
    const XML_PATH_REGION_REQUIRED            = 'mageworx_shipping_calculator/main/region_required';
    const XML_PATH_CALCULATE_BY_ZIP           = 'mageworx_shipping_calculator/main/calculate_by_zip';
    const XML_PATH_ZIP_REQUIRED               = 'mageworx_shipping_calculator/main/zip_required';
    const XML_PATH_HIGHLIGHT_CHEAPEST_METHOD  = 'mageworx_shipping_calculator/main/highlight_cheapest_method';
    const XML_PATH_CALCULATOR_TITLE           = 'mageworx_shipping_calculator/main/title';
    const XML_PATH_CALCULATOR_DESCRIPTION     = 'mageworx_shipping_calculator/main/description';
    const XML_PATH_CALCULATOR_NOT_FOUND_ERROR = 'mageworx_shipping_calculator/main/not_found_error';

    const XML_PATH_CALCULATOR_PRODUCT_RESTRICTIONS  = 'mageworx_shipping_calculator/restrictions/ignored_products';
    const XML_PATH_CALCULATOR_CATEGORY_RESTRICTIONS = 'mageworx_shipping_calculator/restrictions/ignored_categories';

    const XML_PATH_SHIPPING_METHODS_CONFIG = 'mageworx_shipping_calculator/shipping_methods/configuration';

    const IMAGE_URL_PATH = '/mageworx/calculator/methods/';

    /**
     * @var \Magento\Framework\Math\Random
     */
    protected $mathRandom;

    /**
     * @var SerializeJson
     */
    protected $serializer;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * Data constructor.
     *
     * @param Context $context
     * @param Random $mathRandom
     * @param SerializeJson $serializer
     * @param StoreManagerInterface $storeManager
     * @param Logger $logger
     */
    public function __construct(
        Context $context,
        Random $mathRandom,
        SerializeJson $serializer,
        StoreManagerInterface $storeManager,
        Logger $logger
    ) {
        $this->mathRandom   = $mathRandom;
        $this->serializer   = $serializer;
        $this->storeManager = $storeManager;
        $this->logger       = $logger;
        parent::__construct($context);
    }

    /**
     * @param int|null $storeId
     * @return bool
     */
    public function isEnabled(int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            static::XML_PATH_ENABLED,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param int|null $storeId
     * @return bool
     */
    public function isCalculateByCountry(int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            static::XML_PATH_CALCULATE_BY_COUNTRY,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param int|null $storeId
     * @return bool
     */
    public function isCountryRequired(int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            static::XML_PATH_COUNTRY_REQUIRED,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param int|null $storeId
     * @return bool
     */
    public function isCalculateByRegion(int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            static::XML_PATH_CALCULATE_BY_REGION,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param int|null $storeId
     * @return bool
     */
    public function isRegionRequired(int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            static::XML_PATH_REGION_REQUIRED,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param int|null $storeId
     * @return bool
     */
    public function isCalculateByZip(int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            static::XML_PATH_CALCULATE_BY_ZIP,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param int|null $storeId
     * @return bool
     */
    public function isZipRequired(int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            static::XML_PATH_ZIP_REQUIRED,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param int|null $storeId
     * @return bool
     */
    public function isCheapestMethodHighlightNeeded(int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            static::XML_PATH_HIGHLIGHT_CHEAPEST_METHOD,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Do calculations on page load
     *
     * @param int|null $storeId
     * @return bool
     */
    public function isCalculateOnLoadEnabled(int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            static::XML_PATH_CALCULATE_ON_LOAD,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Get title for the calculator header
     *
     * @param int|null $storeId
     * @return string
     */
    public function getCalculatorTitle(int $storeId = null): string
    {
        return $this->scopeConfig->getValue(
            static::XML_PATH_CALCULATOR_TITLE,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Get description for the calculator
     *
     * @param int|null $storeId
     * @return string
     */
    public function getCalculatorDescription(int $storeId = null): string
    {
        return $this->scopeConfig->getValue(
            static::XML_PATH_CALCULATOR_DESCRIPTION,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Get error message when no rates available
     *
     * @param int|null $storeId
     * @return string
     */
    public function getCalculatorNotFoundErrorMessage(int $storeId = null): string
    {
        return $this->scopeConfig->getValue(
            static::XML_PATH_CALCULATOR_NOT_FOUND_ERROR,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Parse comma separated string of ids and convert it to array ( int[] )
     *
     * @param string $string
     * @return array
     */
    private function convertCommaSeparatedStringToIntArray(string $string): array
    {
        $result = explode(',', $string);
        array_walk(
            $result,
            function (&$value) {
                $value = (int)trim($value);
            }
        );
        $result = array_filter($result);

        return $result;
    }

    /**
     * Get category ids for which calculator should be disabled
     *
     * @param int|null $storeId
     * @return array
     */
    public function getIgnoredCategoryIds(int $storeId = null): array
    {
        $commaSeparatedIds = $this->scopeConfig->getValue(
            static::XML_PATH_CALCULATOR_CATEGORY_RESTRICTIONS,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );

        $result = [];
        if ($commaSeparatedIds) {
            $result = $this->convertCommaSeparatedStringToIntArray($commaSeparatedIds);
        }

        return $result;
    }

    /**
     * Get product ids for which calculator should be disabled
     *
     * @param int|null $storeId
     * @return array
     */
    public function getIgnoredProductIds(int $storeId = null): array
    {
        $commaSeparatedIds = $this->scopeConfig->getValue(
            static::XML_PATH_CALCULATOR_PRODUCT_RESTRICTIONS,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );

        $result = [];
        if ($commaSeparatedIds) {
            $result = $this->convertCommaSeparatedStringToIntArray($commaSeparatedIds);
        }

        return $result;
    }

    /**
     * Get default country id from the tax configuration of Magento
     *
     * @param int|null $storeId
     * @return string|null
     */
    public function getTaxDefaultCountryId(int $storeId = null): ?string
    {
        return $this->scopeConfig->getValue(
            TaxConfig::CONFIG_XML_PATH_DEFAULT_COUNTRY,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Get default region\region_id from the tax configuration of Magento
     *
     * @param int|null $storeId
     * @return string|null
     */
    public function getTaxDefaultRegion(int $storeId = null): ?string
    {
        return $this->scopeConfig->getValue(
            TaxConfig::CONFIG_XML_PATH_DEFAULT_REGION,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Get default postcode from the tax configuration of Magento
     *
     * @param int|null $storeId
     * @return string|null
     */
    public function getTaxDefaultPostcode(int $storeId = null): ?string
    {
        return $this->scopeConfig->getValue(
            TaxConfig::CONFIG_XML_PATH_DEFAULT_POSTCODE,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param int|null $storeId
     * @return array
     */
    public function getShippingMethodsConfiguration(int $storeId = null): array
    {
        $result = $this->scopeConfig->getValue(
            static::XML_PATH_SHIPPING_METHODS_CONFIG,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );

        if (empty($result)) {
            return [];
        }

        if (is_string($result)) {
            $result = $this->unserializeValue($result);
        }

        foreach ($result as $methodCode => $values) {
            if (empty($values['image'])) {
                continue;
            }

            try {
                $fullImageUrl                 = $this->getBaseImagePath($storeId) . $values['image'];
                $result[$methodCode]['image'] = $fullImageUrl;
            } catch (LocalizedException $exception) {
                $result[$methodCode]['image'] = '';
            }
        }

        return $result;
    }

    /** FOR METHODS SETTING IN STORE CONFIG */

    /**
     * Create a value from a storable representation
     *
     * @param int|float|string $value
     * @return array
     */
    public function unserializeValue($value): array
    {
        if (is_string($value) && !empty($value)) {
            try {
                return $this->serializer->unserialize($value);
            } catch (\Exception $e) {
                $this->logger->critical($e->getMessage());

                return [];
            }
        } else {
            return [];
        }
    }

    /**
     * Generate a storable representation of a value
     *
     * @param int|float|string|array $value
     * @return string
     */
    public function serializeValue($value)
    {
        if (is_array($value)) {
            $data = [];
            foreach ($value as $methodId => $title) {
                if (!array_key_exists($methodId, $data)) {
                    $data[$methodId] = $title;
                }
            }

            try {
                return $this->serializer->serialize($data);
            } catch (\Exception $e) {
                $this->logger->critical($e->getMessage());

                return '';
            }
        } else {
            return '';
        }
    }

    /**
     * Check whether value is in form retrieved by _encodeArrayFieldValue()
     *
     * @param string|array $value
     * @return bool
     */
    public function isEncodedArrayFieldValue($value): bool
    {
        if (!is_array($value)) {
            return false;
        }

        unset($value['__empty']);
        foreach ($value as $row) {
            if (!is_array($row)
                || !array_key_exists('methods_id', $row)
            ) {
                return false;
            }
        }

        return true;
    }

    /**
     * Encode value to be used in \Magento\Config\Block\System\Config\Form\Field\FieldArray\AbstractFieldArray
     *
     * @param array $value
     * @return array
     * @throws LocalizedException
     */
    public function encodeArrayFieldValue(array $value): array
    {
        $result = [];
        foreach ($value as $methodId => $data) {
            if (!is_array($data)) {
                continue;
            }
            $resultId          = $this->mathRandom->getUniqueHash('_');
            $result[$resultId] = $data;
        }

        return $result;
    }

    /**
     * Decode value from used in \Magento\Config\Block\System\Config\Form\Field\FieldArray\AbstractFieldArray
     *
     * @param array $value
     * @return array
     */
    public function decodeArrayFieldValue(array $value): array
    {
        $result = [];
        unset($value['__empty']);
        foreach ($value as $row) {
            if (!is_array($row)
                || !array_key_exists('methods_id', $row)
            ) {
                continue;
            }
            $methodId          = $row['methods_id'];
            $result[$methodId] = $row;
        }

        return $result;
    }

    /**
     * Returns base url for images
     *
     * @param int|null $storeId
     * @return string
     * @throws NoSuchEntityException
     */
    public function getBaseImagePath(int $storeId = null): string
    {
        $store        = $this->storeManager->getStore($storeId);
        $baseMediaUrl = $store->getBaseUrl(UrlInterface::URL_TYPE_MEDIA);

        $fullBaseUrl = $baseMediaUrl . static::IMAGE_URL_PATH;

        return $fullBaseUrl;
    }
}
