<?php
/**
 * Copyright © 2016 MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\OptionLink\Model;

use Magento\Catalog\Model\ResourceModel\Product\Option\Value\CollectionFactory as MagentoValueFactory;
use Magento\Framework\ObjectManagerInterface as ObjectManager;
use Magento\Store\Model\StoreManagerInterface as StoreManager;
use MageWorx\OptionTemplates\Model\ResourceModel\Group\Option\Value\CollectionFactory as MageWorxValueFactory;

/**
 * Class OptionValue.
 */
class OptionValue
{
    protected MageWorxValueFactory $mageworxValueFactory;
    protected MagentoValueFactory  $magentoValueFactory;
    protected StoreManager         $storeManager;
    protected ?ObjectManager       $objectManager = null;

    /**
     * OptionValue constructor.
     *
     * @param StoreManager $storeManager
     * @param ObjectManager $objectManager
     */
    public function __construct(
        MageWorxValueFactory $mageworxValueFactory,
        MagentoValueFactory  $magentoValueFactory,
        StoreManager         $storeManager,
        ObjectManager        $objectManager
    ) {
        $this->mageworxValueFactory = $mageworxValueFactory;
        $this->magentoValueFactory  = $magentoValueFactory;
        $this->storeManager         = $storeManager;
        $this->objectManager        = $objectManager;
    }

    /**
     * Update option values to original data before option save.
     *
     * @param array $currentValues
     * @param array $originalValues
     * @param array $fields
     * @return array
     */
    public function updateOptionValuesBeforeSave(array $currentValues, array $originalValues, array $fields): array
    {
        $updatedValues = [];

        foreach ($currentValues as $vKey => $currentValue) {
            $id = $currentValue['option_type_id'] ?? $vKey;

            // If the originalValue is missing - do not update
            if (!isset($originalValues[$id])) {
                $updatedValues[$vKey] = $currentValue;
                continue;
            }

            // If there is no SKU or it is invalid - skip it
            if (empty($currentValue['sku']) || empty($currentValue['sku_is_valid'])) {
                $updatedValues[$vKey] = $currentValue;
                continue;
            }

            $originalValue = $originalValues[$id];

            foreach ($fields as $field) {
                if (isset($originalValue[$field])) {
                    $currentValue[$field] = $originalValue[$field];
                }
            }

            $updatedValues[$vKey] = $currentValue;
        }

        return $updatedValues;
    }

    /**
     * Retrieve original option value data by options ids
     *
     * @param bool $isProductSave
     * @param int|array $optionIds
     * @return array
     */
    public function loadOriginalOptions($optionIds, $isProductSave = true)
    {
        $options = [];
        $storeId = $this->storeManager->getStore()->getId();

        if ($isProductSave) { // we save product
            $valuesCollection = $this->magentoValueFactory->create();
        } else { // we save options template
            $valuesCollection = $this->mageworxValueFactory->create();
        }

        $valuesCollection
            ->addTitleToResult($storeId)
            ->addPriceToResult($storeId);

        $valuesCollection->getSelect()
                         ->where('option_id IN (?)', $optionIds);

        $sql = $valuesCollection->getSelect();

        $resource   = $this->objectManager->get('Magento\Framework\App\ResourceConnection');
        $connection = $resource->getConnection();

        $options = $connection->fetchAssoc($sql);

        return $options;
    }
}
