<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types = 1);

namespace Magento\FunctionalTestingFramework\Console;

use Magento\FunctionalTestingFramework\Exceptions\TestFrameworkException;
use Magento\FunctionalTestingFramework\Util\Path\FilePathFormatter;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;

class CleanProjectCommand extends Command
{
    private const SUCCESS_EXIT_CODE = 0;

    /**
     * Configures the current command.
     *
     * @return void
     */
    protected function configure()
    {
        $this->setName('reset')
            ->setDescription(
                'This command will clean any configuration files from the environment (not including .env), as well as generated artifacts.' // phpcs:ignore
            )
            ->addOption('hard', null, InputOption::VALUE_NONE, "parameter to force reset of configuration files.");
    }

    /**
     * Executes the current command.
     *
     * @param InputInterface  $input
     * @param OutputInterface $output
     * @return integer
     * @throws \Symfony\Component\Console\Exception\LogicException
     * @throws TestFrameworkException
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $configFiles = [
            // codeception.yml file for top level config
            FilePathFormatter::format(TESTS_BP) . 'codeception.yml',
            // functional.suite.yml for test execution config
            FilePathFormatter::format(TESTS_BP) . 'tests' . DIRECTORY_SEPARATOR . 'functional.suite.yml',
            // Acceptance Tester Actions generated by codeception
            FilePathFormatter::format(FW_BP) . 'src/Magento/FunctionalTestingFramework/_generated',
            // AcceptanceTester Class generated by codeception
            FilePathFormatter::format(FW_BP) . 'src/Magento/FunctionalTestingFramework/AcceptanceTester.php'
        ];

        $generatedFiles = [
            FilePathFormatter::format(TESTS_MODULE_PATH) . '_generated'
        ];

        $isHardReset = $input->getOption('hard');
        $fileSystem = new Filesystem();
        $finder = new Finder();
        $finder->files()->name('*.php')->in(
            realpath(FilePathFormatter::format(FW_BP) . 'src/Magento/FunctionalTestingFramework/Group/')
        );
        $filesForRemoval = [];

        // include config files if user specifies a hard reset for deletion
        if ($isHardReset) {
            $filesForRemoval = array_merge($filesForRemoval, $configFiles);
        }

        // include the files mftf generates during test execution in TESTS_BP for deletion
        $filesForRemoval = array_merge($filesForRemoval, $generatedFiles);

        if ($output->isVerbose()) {
            $output->writeln('Deleting Files:');
        }

        // delete any suite based group files
        foreach ($finder->files() as $file) {
            if ($output->isVerbose()) {
                $output->writeln($file->getRealPath());
            }

            $fileSystem->remove($file);
        }

        // delete files specified for removal
        foreach ($filesForRemoval as $fileForRemoval) {
            if ($fileSystem->exists($fileForRemoval) && $output->isVerbose()) {
                $output->writeln($fileForRemoval);
            }

            $fileSystem->remove($fileForRemoval);
        }

        $output->writeln('mftf files removed from filesystem.');

        return self::SUCCESS_EXIT_CODE;
    }
}
