<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */
declare(strict_types = 1);

namespace MageWorx\ShippingRules\Test\Unit\ArtificialCarrier;

use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\TestFramework\Unit\Helper\ObjectManager;
use Magento\Quote\Model\Quote\Address\RateRequest;
use Magento\Quote\Model\Quote\Address\RateResult\Method;
use Magento\Quote\Model\Quote\Address\RateResult\MethodFactory;
use Magento\Shipping\Model\Rate\Result;
use Magento\Shipping\Model\Rate\ResultFactory;
use MageWorx\ShippingRules\Api\CartPriceCalculationConfigInterface;
use MageWorx\ShippingRules\Api\CartPriceCalculationConfigInterfaceFactory;
use MageWorx\ShippingRules\Api\CartPriceCalculationInterface;
use MageWorx\ShippingRules\Model\Carrier;
use MageWorx\ShippingRules\Model\Carrier\Artificial;
use MageWorx\ShippingRules\Model\Carrier\Method\Rate as MageWorxRate;
use MageWorx\ShippingRules\Model\CarrierFactory;
use MageWorx\ShippingRules\Model\ResourceModel\Rate\Collection;
use MageWorx\ShippingRules\Model\ResourceModel\Rate\CollectionFactory;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use function array_merge;
use function get_class_methods;

class CollectRatesTest extends TestCase
{
    public const TESTED_CLASS_NAME = 'MageWorx\ShippingRules\Model\Carrier\Artificial';

    /**
     * @var Artificial
     */
    protected $model;

    /**
     * @var RateRequest|MockObject
     */
    protected $rateRequestMock;

    /**
     * @var CarrierFactory|MockObject
     */
    protected $carrierFactoryMock;

    /**
     * @var Carrier|MockObject
     */
    protected $carrierMock;

    /**
     * @var ResultFactory|MockObject
     */
    protected $rateResultFactoryMock;

    /**
     * @var Result|MockObject
     */
    protected $rateResultMock;

    /**
     * @var Carrier\Method|MockObject
     */
    protected $methodMock1;

    /**
     * @var Carrier\Method|MockObject
     */
    protected $methodMock2;

    /**
     * @var MethodFactory|MockObject
     */
    protected $rateMethodFactoryMock;

    /**
     * @var Method|MockObject
     */
    protected $rateMethodMock;

    /**
     * @var CollectionFactory|MockObject
     */
    protected $rateCollectionFactoryMock;

    /**
     * @var Collection|MockObject
     */
    protected $rateCollectionMock;

    /**
     * @var MageWorxRate|MockObject
     */
    protected $rateOneMock;

    /**
     * @var CartPriceCalculationConfigInterfaceFactory|MockObject
     */
    protected $cartPriceCalcConfigFactoryMock;

    /**
     * @var CartPriceCalculationConfigInterface|MockObject
     */
    protected $cartPriceCalcConfigMock;

    /**
     * @var CartPriceCalculationInterface|MockObject
     */
    protected $cartPriceCalculationMock;

    protected $objectManager;

    /**
     * @return void
     */
    public function setUp(): void
    {
        $this->objectManager = new ObjectManager($this);

        $methods = array_merge(
            get_class_methods(RateRequest::class)
        );

        $this->rateRequestMock = $this->getMockBuilder(RateRequest::class)
                                      ->onlyMethods($methods)
                                      ->addMethods(['getAllItems', 'getStoreId'])
                                      ->disableOriginalConstructor()
                                      ->getMock();

        $this->rateResultFactoryMock = $this->getMockBuilder('\Magento\Shipping\Model\Rate\ResultFactory')
                                            ->disableOriginalConstructor()
                                            ->getMock();

        $this->rateResultMock = $this->getMockBuilder(Result::class)
                                     ->addMethods(['getMethodTitle'])
                                     ->disableOriginalConstructor()
                                     ->getMock();

        $this->carrierFactoryMock = $this->getMockBuilder('\MageWorx\ShippingRules\Model\CarrierFactory')
                                         ->disableOriginalConstructor()
                                         ->getMock();

        $this->carrierMock = $this->getMockBuilder(Carrier::class)
                                  ->disableOriginalConstructor()
                                  ->getMock();

        $methodMethods     = array_merge(
            get_class_methods(Carrier\Method::class)
        );
        $this->methodMock1 = $this->getMockBuilder(Carrier\Method::class)
                                  ->onlyMethods($methodMethods)
                                  ->addMethods(['getMethodTitle'])
                                  ->disableOriginalConstructor()
                                  ->getMock();

        $this->methodMock2 = $this->getMockBuilder(Carrier\Method::class)
                                  ->onlyMethods($methodMethods)
                                  ->addMethods(['getMethodTitle'])
                                  ->disableOriginalConstructor()
                                  ->getMock();

        $this->rateMethodFactoryMock = $this->getMockBuilder(
            '\Magento\Quote\Model\Quote\Address\RateResult\MethodFactory'
        )
                                            ->disableOriginalConstructor()
                                            ->getMock();

        $this->rateMethodMock = $this->getMockBuilder(Method::class)
                                     ->disableOriginalConstructor()
                                     ->getMock();

        $this->rateCollectionFactoryMock = $this->getMockBuilder(
            '\MageWorx\ShippingRules\Model\ResourceModel\Rate\CollectionFactory'
        )
                                                ->disableOriginalConstructor()
                                                ->getMock();

        $this->rateCollectionMock = $this->getMockBuilder(
            Collection::class
        )
                                         ->disableOriginalConstructor()
                                         ->getMock();

        $this->rateOneMock = $this->getMockBuilder(MageWorxRate::class)
                                  ->disableOriginalConstructor()
                                  ->getMock();

        $this->cartPriceCalcConfigFactoryMock = $this->getMockBuilder(
            '\MageWorx\ShippingRules\Api\CartPriceCalculationConfigInterfaceFactory'
        )
                                                     ->disableOriginalConstructor()
                                                     ->getMock();

        $this->cartPriceCalcConfigMock = $this->getMockBuilder(
            CartPriceCalculationConfigInterface::class
        )
                                              ->disableOriginalConstructor()
                                              ->getMock();

        $this->cartPriceCalculationMock = $this->getMockBuilder(
            CartPriceCalculationInterface::class
        )
                                               ->disableOriginalConstructor()
                                               ->getMock();
        $args                           = [
            'carrierFactory'             => $this->carrierFactoryMock,
            'rateResultFactory'          => $this->rateResultFactoryMock,
            'rateMethodFactory'          => $this->rateMethodFactoryMock,
            'rateCollectionFactory'      => $this->rateCollectionFactoryMock,
            'cartPriceCalcConfigFactory' => $this->cartPriceCalcConfigFactoryMock,
            'cartPriceCalculation'       => $this->cartPriceCalculationMock
        ];

        /** @var Artificial model */
        $this->model = $this->objectManager->getObject(
            self::TESTED_CLASS_NAME,
            $args
        );
    }

    /**
     * @return void
     * @throws NoSuchEntityException
     */
    public function testCollectRatesWithEmptyMethods()
    {
        $requestItems = [];
        $storeId      = 300;

        $this->rateRequestMock->expects($this->atLeastOnce())
                              ->method('getAllItems')
                              ->willReturn($requestItems);

        $this->rateRequestMock->expects($this->atLeastOnce())
                              ->method('getStoreId')
                              ->willReturn($storeId);

        $this->rateResultFactoryMock->expects($this->atLeastOnce())
                                    ->method('create')
                                    ->willReturn($this->rateResultMock);

        $this->carrierFactoryMock->expects($this->atLeastOnce())
                                 ->method('create')
                                 ->willReturn($this->carrierMock);

        $this->carrierMock->expects($this->atLeastOnce())
                          ->method('load')
                          ->withAnyParameters()
                          ->willReturnSelf();

        $carrierData = [];
        $methods     = [];

        $this->carrierMock->expects($this->atLeastOnce())
                          ->method('getData')
                          ->willReturn($carrierData);

        $this->carrierMock->expects($this->atLeastOnce())
                          ->method('getMethods')
                          ->willReturn($methods);

        $this->model->collectRates($this->rateRequestMock);
    }

    /**
     * @return void
     * @throws NoSuchEntityException
     */
    public function testCollectRates()
    {
        $requestItems = [];
        $storeId      = 300;

        $this->rateRequestMock->expects($this->atLeastOnce())
                              ->method('getAllItems')
                              ->willReturn($requestItems);

        $this->rateRequestMock->expects($this->atLeastOnce())
                              ->method('getStoreId')
                              ->willReturn($storeId);

        $this->rateResultFactoryMock->expects($this->atLeastOnce())
                                    ->method('create')
                                    ->willReturn($this->rateResultMock);

        $this->carrierFactoryMock->expects($this->atLeastOnce())
                                 ->method('create')
                                 ->willReturn($this->carrierMock);

        $this->carrierMock->expects($this->atLeastOnce())
                          ->method('load')
                          ->withAnyParameters()
                          ->willReturnSelf();

        $carrierData = [];
        $methods     = [
            $this->methodMock1,
            $this->methodMock2
        ];

        $this->carrierMock->expects($this->atLeastOnce())
                          ->method('getData')
                          ->willReturn($carrierData);

        $this->carrierMock->expects($this->atLeastOnce())
                          ->method('getMethods')
                          ->willReturn($methods);

        // Method 1 will be skipped as inactive
        $this->methodMock1->expects($this->once())
                          ->method('getActive')
                          ->willReturn(false);

        // The only active method with rates
        $this->methodMock2->expects($this->once())
                          ->method('getActive')
                          ->willReturn(true);

        // Calculate available rates by max priority
        $this->methodMock2->expects($this->atLeast(2))
                          ->method('getMultipleRatesPrice')
                          ->willReturn(MageWorxRate::MULTIPLE_RATES_PRICE_CALCULATION_MAX_PRIORITY);

        $this->rateMethodFactoryMock->expects($this->once())
                                    ->method('create')
                                    ->willReturn($this->rateMethodMock);

        $this->rateCollectionFactoryMock->expects($this->once())
                                        ->method('create')
                                        ->willReturn($this->rateCollectionMock);

        $this->rateOneMock->expects($this->any())
                          ->method('getId')
                          ->willReturn(1);
        $this->rateOneMock->expects($this->any())
                          ->method('applyRateToMethod')
                          ->with($this->rateMethodMock)
                          ->willReturn($this->rateMethodMock);

        $rates = [
            $this->rateOneMock
        ];

        $this->rateCollectionMock->expects($this->once())
                                 ->method('getItems')
                                 ->willReturn($rates);

        $this->cartPriceCalcConfigFactoryMock->expects($this->once())
                                             ->method('create')
                                             ->willReturn($this->cartPriceCalcConfigMock);

        $basePrice = 0.0;
        $this->cartPriceCalculationMock->expects($this->once())
                                       ->method('calculateBasePrice')
                                       ->willReturn($basePrice);

        $this->model->collectRates($this->rateRequestMock);
    }
}
