<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingRules\Model\Plugin\Address;

use Magento\Quote\Model\Quote\Address as QuoteAddress;
use MageWorx\ShippingRules\Helper\Data as Helper;

class SortShippingRates
{
    protected Helper $helper;

    public function __construct(
        Helper $helper
    ) {
        $this->helper = $helper;
    }

    /**
     * Sort shipping carriers by sort order and price
     *
     * @param QuoteAddress $subject
     * @param array $result
     * @return array
     */
    public function afterGetGroupedAllShippingRates(
        QuoteAddress $subject,
        array        $result
    ) {
        if ($this->helper->isNeedToSortCarriers()) {
            $this->sortRatesBySortOrder($result);
        }

        if ($this->helper->displayCheapestRateAtTop()) {
            $this->sortRatesByMinPrice($result);
        }

        return $result;
    }

    /**
     * Sort shipping rates by sort order
     *
     * @param array $result The array of shipping rates to be sorted
     * @return void
     */
    private function sortRatesBySortOrder(array &$result): void
    {
        uasort(
            $result,
            function ($first, $second) {
                if (!isset($first[0]) || !isset($second[0])) {
                    // If there are no elements to compare - return 0 to leave the order unchanged
                    return 0;
                }

                if (isset($first[0]->carrier_sort_order) && isset($second[0]->carrier_sort_order)) {
                    return $first[0]->carrier_sort_order > $second[0]->carrier_sort_order ? -1 : 1;
                }

                // If there is no property - return 0 to leave the order unchanged
                return 0;
            }
        );
    }

    /**
     * Sorts the rates in the given result array by minimum price from low to high
     *
     * @param array $result
     * @return void
     */
    private function sortRatesByMinPrice(array &$result): void
    {
        $carrierMinPrice = [];
        foreach ($result as $rates) {
            foreach ($rates as $rate) {
                $ratePrice = $rate->getPrice();
                if (isset($carrierMinPrice[$rate->getCarrier()]) && $carrierMinPrice[$rate->getCarrier()] < $ratePrice) {
                    continue;
                }
                $carrierMinPrice[$rate->getCarrier()] = $ratePrice;
            }
        }

        uasort(
            $result,
            function ($first, $second) use ($carrierMinPrice) {
                if (!isset($first[0]) || !isset($second[0])) {
                    // If there are no elements to compare - return 0 to leave the order unchanged
                    return 0;
                }

                return $carrierMinPrice[$first[0]->getCarrier()] >= $carrierMinPrice[$second[0]->getCarrier()] ? 1 : -1;
            }
        );
    }
}
