<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingCalculatorBase\Block\Product\View\Calculator;

use Magento\Checkout\Block\Checkout\LayoutProcessorInterface;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Directory\Model\Config\Source\AllRegion;
use Magento\Directory\Model\Config\Source\Country;
use Magento\Directory\Model\ResourceModel\Country\CollectionFactory as CountryCollectionFactory;
use Magento\Directory\Model\ResourceModel\Region\CollectionFactory;
use Magento\Framework\UrlInterface;
use MageWorx\ShippingCalculatorBase\Api\AddressResolverInterface;
use MageWorx\ShippingCalculatorBase\Helper\Data as Helper;

/**
 * Class FormLayoutProcessor
 */
class FormLayoutProcessor implements LayoutProcessorInterface
{
    /**
     * @var Country
     */
    private $countryOptions;

    /**
     * @var AllRegion
     */
    private $regionsOptions;

    /**
     * @var CountryCollectionFactory
     */
    private $countryCollectionFactory;

    /**
     * @var CollectionFactory
     */
    private $regionsCollectionFactory;

    /**
     * @var UrlInterface
     */
    private $url;

    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @var CustomerSession
     */
    private $customerSession;

    /**
     * @var Helper
     */
    private $helper;

    /**
     * @var AddressResolverInterface
     */
    private $addressResolver;

    /**
     * FormLayoutProcessor constructor.
     *
     * @param CountryCollectionFactory $countryCollectionFactory
     * @param CollectionFactory $regionsCollectionFactory
     * @param UrlInterface $url
     * @param CheckoutSession $checkoutSession
     * @param CustomerSession $customerSession
     * @param AddressResolverInterface $addressResolver
     * @param Helper $helper
     */
    public function __construct(
        CountryCollectionFactory $countryCollectionFactory,
        CollectionFactory $regionsCollectionFactory,
        UrlInterface $url,
        CheckoutSession $checkoutSession,
        CustomerSession $customerSession,
        AddressResolverInterface $addressResolver,
        Helper $helper
    ) {
        $this->countryCollectionFactory = $countryCollectionFactory;
        $this->regionsCollectionFactory = $regionsCollectionFactory;

        $this->url             = $url;
        $this->checkoutSession = $checkoutSession;
        $this->customerSession = $customerSession;
        $this->addressResolver = $addressResolver;
        $this->helper          = $helper;
    }

    /**
     * Add country options, region settings etc.
     *
     * @param array $jsLayout
     * @return array
     */
    public function process($jsLayout)
    {
        $this->processAddressFields($jsLayout);

        if (isset($jsLayout["components"]["shipping-calculator"])) {
            $jsLayout["components"]["shipping-calculator"]["title"] = $this->helper->getCalculatorTitle();
        }

        if (isset($jsLayout["components"]["shipping-calculator"]["children"]["description"])) {
            $jsLayout["components"]["shipping-calculator"]["children"]
            ["description"]["description"] = $this->helper->getCalculatorDescription();
        }

        if (isset($jsLayout["components"]["shipping-calculator"]["children"]["estimateForm"])) {
            $jsLayout["components"]["shipping-calculator"]["children"]["estimateForm"]['refreshUrl'] =
                $this->url->getUrl('mageworx_shipping_calculator/calculator/refresh');
        }

        $this->addDefaultAddress($jsLayout);

        if (isset($jsLayout["components"]["shipping-calculator"]["children"]["methods-list"])) {
            $jsLayout["components"]["shipping-calculator"]["children"]["methods-list"]["displayCheapestPrice"] =
                $this->helper->isCheapestMethodHighlightNeeded();
        }

        if ($this->helper->isCalculateOnLoadEnabled()) {
            $jsLayout["components"]["shipping-calculator"]["children"]["estimateForm"]
            ["calculateShippingMethodsOnLoad"] = true;
        }

        return $jsLayout;
    }

    /**
     * Fill address fields configuration: option, status etc.
     *
     * @param array $jsLayout
     */
    private function processAddressFields(array &$jsLayout): void
    {
        if (!isset($jsLayout["components"]["shipping-calculator"]["children"]["estimateForm"]["children"]["fieldset"])
        ) {
            return;
        }

        $fieldset = &$jsLayout["components"]["shipping-calculator"]["children"]["estimateForm"]["children"]["fieldset"];

        if (isset($fieldset["children"]["country_id"])) {
            $countryOptions                                          = $this->getCountryOptions();
            $fieldset["children"]["country_id"]["config"]["options"] = $countryOptions;
            if (!$this->helper->isCalculateByCountry()) {
                $fieldset["children"]["country_id"]["disabled"] = true;
                $fieldset["children"]["country_id"]["visible"]  = false;
            }
        }

        if (isset($fieldset["children"]["region_id"])) {
            $regionOptions                                          = $this->getRegionsOptions();
            $fieldset["children"]["region_id"]["config"]["options"] = $regionOptions;
            if (!empty($regionOptions)) {
                $firstOption = reset($regionOptions);
                $caption = empty($firstOption['value']) ? '' : $firstOption['value'];

                $fieldset["children"]["region_id"]["config"]["captionValue"] = $caption;
            }
            if (!$this->helper->isCalculateByRegion()) {
                $fieldset["children"]["region_id"]["disabled"] = true;
                $fieldset["children"]["region"]["disabled"]    = true;
                $fieldset["children"]["region_id"]["visible"]  = false;
                $fieldset["children"]["region"]["visible"]     = false;
            }
            if ($this->helper->isCalculateByCountry()) {
                $fieldset["children"]["region_id"]["required"] = $this->helper->isRegionRequired();
                $fieldset["children"]["region_id"]["config"]["requiredEntry"] = $this->helper->isRegionRequired();
                $fieldset["children"]["region"]["required"] = $this->helper->isRegionRequired();
                $fieldset["children"]["region"]["validation"]["required-entry"] = $this->helper->isRegionRequired();
            }
        }

        if (isset($fieldset["children"]["postcode"])) {
            if (!$this->helper->isCalculateByZip()) {
                $fieldset["children"]["postcode"]["disabled"] = true;
                $fieldset["children"]["postcode"]["visible"]  = false;
            }
            $fieldset["children"]["postcode"]["required"] = $this->helper->isZipRequired();
            $fieldset["children"]["postcode"]["skipValidation"] = !$this->helper->isZipRequired();
        }
    }

    /**
     * @param array $jsLayout
     */
    private function addDefaultAddress(array &$jsLayout): void
    {
        $fieldset = &$jsLayout["components"]["shipping-calculator"]["children"]["estimateForm"]["children"]["fieldset"];

        $countryId = $this->addressResolver->getCountryId();
        if (isset($fieldset["children"]["country_id"]) && $countryId) {
            $fieldset["children"]["country_id"]["default"] = $countryId;
        }

        $region = $this->addressResolver->getRegionId();
        if (isset($fieldset["children"]["region"]) && $region) {
            $fieldset["children"]["region"]["default"] = $region;
        }

        $regionId = $this->addressResolver->getRegionId();
        if (isset($fieldset["children"]["region_id"]) && $regionId) {
            $fieldset["children"]["region_id"]["default"] = $regionId;
        }

        $postcode = $this->addressResolver->getZipCode();
        if (isset($fieldset["children"]["postcode"]) && $postcode) {
            $fieldset["children"]["postcode"]["default"] = $postcode;
        }
    }

    /**
     * @return array
     */
    private function getCountryOptions(): array
    {
        if (empty($this->countryOptions)) {
            $this->countryOptions = $this->countryCollectionFactory->create()
                                                                   ->loadByStore()
                                                                   ->toOptionArray();
        }

        return $this->countryOptions;
    }

    /**
     * @return array
     */
    private function getRegionsOptions(): array
    {
        if (empty($this->regionsOptions)) {
            $options = $this->regionsCollectionFactory->create()
                                                      ->addAllowedCountriesFilter()
                                                      ->toOptionArray();

            if (count($options) > 0) {
                $caption = reset($options);
                if (isset($caption['value']) && $caption['value'] !== '') {
                    array_unshift(
                        $options,
                        ['title' => '', 'value' => '', 'label' => __('Please select a region, state or province.')]
                    );
                }
            }

            $this->regionsOptions = $options;
        }

        return $this->regionsOptions;
    }
}
