<?php
/**
 * Copyright © 2016 MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\OptionLink\Plugin;

use Magento\Catalog\Api\Data\ProductCustomOptionValuesInterface;
use Magento\Framework\Registry;
use Magento\Store\Model\StoreManagerInterface;
use MageWorx\OptionLink\Helper\Attribute as HelperAttribute;
use MageWorx\OptionLink\Model\OptionValue as ModelOptionValue;

/**
 * Class BeforeSaveValues. Update option values data linked by SKU to original values data.
 * We save data linked by SKU when unlink option value only.
 * This plugin we use when Product saving.
 */
class BeforeSaveValues
{
    protected HelperAttribute  $helperAttribute;
    protected ModelOptionValue $modelOptionValue;

    /**
     * Store manager
     *
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected StoreManagerInterface $storeManager;

    /**
     * Magento register
     *
     * @var \Magento\Framework\Registry
     */
    protected Registry $registry;

    protected ?ProductCustomOptionValuesInterface $processAfterSaveValue     = null;
    protected bool                                $afterSaveUpdateInProgress = false;

    /**
     * BeforeSaveValues constructor.
     *
     * @param HelperAttribute $helperAttribute
     * @param ModelOptionValue $modelOptionValue
     * @param StoreManagerInterface $storeManager
     * @param Registry $registry
     */
    public function __construct(
        HelperAttribute       $helperAttribute,
        ModelOptionValue      $modelOptionValue,
        StoreManagerInterface $storeManager,
        Registry              $registry
    ) {
        $this->helperAttribute  = $helperAttribute;
        $this->modelOptionValue = $modelOptionValue;
        $this->storeManager     = $storeManager;
        $this->registry         = $registry;
    }

    /**
     * @param ProductCustomOptionValuesInterface $value
     * @return void
     */
    public function beforeSaveValues(ProductCustomOptionValuesInterface $value): void
    {
        if ($this->afterSaveUpdateInProgress === true) {
            $this->afterSaveUpdateInProgress = false;
            return;
        }

        // Update when Product saving.
        // For OptionTemplates saving we use UpdateOptionValuesBeforeGroupSave observer.
        if ($value instanceof \MageWorx\OptionTemplates\Model\Group\Option\Value) {
            return;
        }

        $originalValues = $this->getOriginalOptions($value);

        if (empty($originalValues)) {
            $this->processAfterSaveValue = $value;
        }

        $currentValues = $value->getValues();
        $fields        = $this->helperAttribute->getConvertedAttributesToFields();

        $currentValues = $this->modelOptionValue
            ->updateOptionValuesBeforeSave($currentValues, $originalValues, $fields);

        $value->setValues($currentValues);
    }

    /**
     * @param ProductCustomOptionValuesInterface $value
     * @return void
     */
    public function afterSaveValues(ProductCustomOptionValuesInterface $value): void
    {
        if ($this->processAfterSaveValue !== null) {
            try {
                $value->save();                                         // Create option_type_id for new option value
                $originalValues = $this->getOriginalOptions($value);    // Requires option_type_id
                $currentValues  = $value->getValues();
                $fields         = $this->helperAttribute->getConvertedAttributesToFields();

                $currentValues = $this->modelOptionValue
                    ->updateOptionValuesBeforeSave($currentValues, $originalValues, $fields);

                $value->setValues($currentValues);
                $this->afterSaveUpdateInProgress = true;
                $this->processAfterSaveValue     = null;
                $value->save(); // Save updated option value with linked product data
            } catch (\Exception $e) {
                $this->processAfterSaveValue = null;
                $this->afterSaveUpdateInProgress = false;
            }
        }
    }

    /**
     * Retrieve original option values data by mageworx options ids.
     * This data stored in 'mageworx_optionlink_original_options' registry
     * created in BeforeDelete plugin.
     *
     * @param ProductCustomOptionValuesInterface $value
     * @return array
     */
    protected function getOriginalOptions(ProductCustomOptionValuesInterface $value): array
    {
        $currentValues = $value->getValues();

        $result = [];
        foreach ($currentValues as $value) {
            if (isset($value['option_type_id'])) {
                $result[$value['option_type_id']] = $value;
            }
        }

        return $result;
    }
}
