<?php
/**
 * Mageplaza
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Mageplaza.com license that is
 * available through the world-wide-web at this URL:
 * https://www.mageplaza.com/LICENSE.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Mageplaza
 * @package     Mageplaza_CustomForm
 * @copyright   Copyright (c) Mageplaza (https://www.mageplaza.com/)
 * @license     https://www.mageplaza.com/LICENSE.txt
 */

namespace Mageplaza\CustomForm\Controller\CustomForm;

use Exception;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\ResponseInterface;
use Magento\Framework\Controller\Result\Redirect;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\HTTP\PhpEnvironment\Request;
use Magento\Store\Model\StoreManagerInterface;
use Mageplaza\CustomForm\Helper\Data;
use Mageplaza\CustomForm\Model\ResourceModel\Responses as ResponsesResource;
use Mageplaza\CustomForm\Model\ResponsesFactory as ResponsesModelFactory;
use Psr\Log\LoggerInterface;

use Magento\Framework\Session\Generic as GenericSession;
use Magento\Framework\Data\Form\FormKey\Validator as FormKeyValidator;

/**
 * Class Submit
 * @package Mageplaza\CustomForm\Controller\CustomForm
 */
class Submit extends Action
{
    /**
     * @var Request
     */
    protected $httpRequest;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var Data
     */
    protected $helperData;

    /**
     * @var ResponsesModelFactory
     */
    protected $responsesFactory;

    /**
     * @var ResponsesResource
     */
    protected $responsesResource;

    protected $customSession;
    protected $formKeyValidator;

    /**
     * Submit constructor.
     *
     * @param Context $context
     * @param Request $httpRequest
     * @param StoreManagerInterface $storeManager
     * @param LoggerInterface $logger
     * @param Data $helperData
     * @param ResponsesModelFactory $responsesFactory
     * @param ResponsesResource $responsesResource
     */
    public function __construct(
        Context $context,
        Request $httpRequest,
        StoreManagerInterface $storeManager,
        LoggerInterface $logger,
        Data $helperData,
        ResponsesModelFactory $responsesFactory,
        ResponsesResource $responsesResource,
        GenericSession $customSession,
        FormKeyValidator $formKeyValidator,
    ) {
        $this->httpRequest = $httpRequest;
        $this->storeManager = $storeManager;
        $this->logger = $logger;
        $this->helperData = $helperData;
        $this->responsesFactory = $responsesFactory;
        $this->responsesResource = $responsesResource;
        $this->customSession = $customSession;
        $this->formKeyValidator = $formKeyValidator;

        parent::__construct($context);
    }

    /**
     * @return ResponseInterface|Redirect|ResultInterface
     */
    public function execute()
    {

        $isValid = true;

        //1. Validate form_key (CSRF)
        if (!$this->formKeyValidator->validate($this->getRequest())) {
            $isValid = false;
            $this->messageManager->addErrorMessage(__('Invalid request.'));
        }

        // 2.Validate honeypot field (should be empty)
        $honeypotValue = $this->getRequest()->getParam('hp_field');
        if (!empty($honeypotValue)) {
            $isValid = false;
            $this->messageManager->addErrorMessage(__('Invalid request.'));
        }


        // 3.Validate secure session token
        $submittedToken = $this->getRequest()->getParam('secure_token');
        $storedToken = $this->customSession->getData('secure_form_token');

        if (!$submittedToken || $submittedToken !== $storedToken) {
            $isValid = false;
            $this->messageManager->addErrorMessage(__('Invalid request.'));
            $this->customSession->setData('secure_form_token', '');
        }

        // 4.Process the form if everything is valid

        $formDataArray = $this->getRequest()->getParam('pages');

        foreach($formDataArray as $data){
            if(isset($data['fieldGroups'])){
                $fieldGroups = $data['fieldGroups'];
                foreach($fieldGroups as $item){
                    if(isset($item['fields'])){
                        $fields = $item['fields'];
                        foreach($fields as $field){
                            $field = $this->cleanInputValue($field);
                            if ($field === '' || $field === null) {
                                $isValid = false;
                                $this->messageManager->addErrorMessage(__('Invalid request.'));
                                break 3;
                            }
                        }
                    }
                }
            }
        }

        if(!$isValid){
            $this->customSession->setData('secure_form_token', '');
            if ($this->getRequest()->isAjax()) {
                return $this->getResponse()->representJson(Data::jsonEncode(['success' => $isValid, 'failureredirect' => $this->_redirect->getRefererUrl()]));
            }
            $resultRedirect = $this->resultRedirectFactory->create();
            return $resultRedirect->setUrl($this->_redirect->getRefererUrl());
        }

        $this->customSession->setData('secure_form_token', '');

        $response = $this->responsesFactory->create();
        /** @var CustomerSession $customerSession */
        $customerSession = $this->helperData->createObject(CustomerSession::class);
        $success = true;
        if ($this->getRequest()->getParam('formId')) {
            try {
                $response->addData([
                    'form_id' => $this->getRequest()->getParam('formId'),
                    'customer_id' => $customerSession->getCustomerId() ?: 0,
                    'store_ids' => $this->storeManager->getStore()->getId(),
                    'store_name' => $this->storeManager->getStore()->getName(),
                    'ip_address' => $this->httpRequest->getClientIp(),
                    'form_data' => Data::jsonEncode($this->getRequest()->getParam('pages')),
                ]);
                $this->responsesResource->save($response);
                $this->messageManager->addSuccessMessage(__('Your form have been submitted successfully '));
            } catch (Exception $e) {
                $success = false;
                $this->logger->critical($e->getMessage());
                $this->messageManager->addErrorMessage(__('Something went wrong while submit form %1'));
            }
        }
        if ($this->getRequest()->isAjax()) {
            return $this->getResponse()->representJson(Data::jsonEncode(['success' => $success, 'successredirect' => $this->_redirect->getRefererUrl()]));
        }
        $resultRedirect = $this->resultRedirectFactory->create();
        $afterSubmitUrl = $this->getRequest()->getParam('afterSubmitUrl');
        $afterSubmitUrl = !empty($afterSubmitUrl) ? $afterSubmitUrl : $this->_redirect->getRefererUrl();

        return $resultRedirect->setUrl($afterSubmitUrl);
    }

    protected function cleanInputValue(string $value): string
    {
        // 1. Trim to avoid weird edge cases
        $cleaned = trim($value);

        // 2. Decode HTML entities like &nbsp; → actual NBSP (U+00A0)
        $cleaned = html_entity_decode($cleaned, ENT_QUOTES | ENT_HTML5, 'UTF-8');

        // 3. Convert literal "&nbsp;" strings (if user literally typed it, not entity)
        $cleaned = str_ireplace(['&nbsp;', '&nbsp', 'nbsp'], ' ', $cleaned);

        // 4. Replace NBSP and zero-width characters with normal space
        $cleaned = preg_replace('/[\x{00A0}\x{200B}-\x{200D}\x{FEFF}]+/u', ' ', $cleaned);

        // 5. Strip HTML tags
        $cleaned = strip_tags($cleaned);

        // 6. Collapse multiple whitespace and trim again
        $cleaned = preg_replace('/\s+/', ' ', $cleaned);
        $cleaned = trim($cleaned);

        return $cleaned;
    }
}
