<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingRules\Model\ResourceModel;

use Magento\Framework\Data\Collection\Db\FetchStrategyInterface;
use Magento\Framework\Data\Collection\EntityFactory;
use Magento\Framework\DataObject;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\DB\Select;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use MageWorx\ShippingRules\Helper\Data;
use Psr\Log\LoggerInterface;
use Zend_Db_Expr;
use Zend_Db_Select_Exception;

/**
 * Class AbstractCollection
 */
abstract class AbstractCollection extends \Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection
{
    /**
     * @var TimezoneInterface
     */
    protected $date;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var array
     */
    protected $_associatedEntitiesMap = [];

    /**
     * @param EntityFactory $entityFactory
     * @param LoggerInterface $logger
     * @param FetchStrategyInterface $fetchStrategy
     * @param ManagerInterface $eventManager
     * @param TimezoneInterface $date
     * @param StoreManagerInterface $storeManager
     * @param Data $helper
     * @param AdapterInterface|null $connection
     * @param AbstractDb|null $resource
     */
    public function __construct(
        EntityFactory          $entityFactory,
        LoggerInterface        $logger,
        FetchStrategyInterface $fetchStrategy,
        ManagerInterface       $eventManager,
        TimezoneInterface      $date,
        StoreManagerInterface  $storeManager,
        Data                   $helper,
        ?AdapterInterface      $connection = null,
        ?AbstractDb            $resource = null
    ) {
        parent::__construct($entityFactory, $logger, $fetchStrategy, $eventManager, $connection, $resource);
        $this->date         = $date;
        $this->storeManager = $storeManager;
        $this->helper       = $helper;
    }

    /**
     * Provide support for store id filter
     *
     * @param string|array $field
     * @param null|string|array $condition
     * @return $this
     */
    public function addFieldToFilter($field, $condition = null)
    {
        if ($field == 'store_ids') {
            return $this->addStoreFilter($condition);
        }

        parent::addFieldToFilter($field, $condition);

        return $this;
    }

    /**
     * Limit entity collection by specific stores
     *
     * @param int|int[]|Store $storeId
     * @return $this
     */
    public function addStoreFilter($storeId)
    {
        $this->joinStoreTable();
        if ($storeId instanceof Store) {
            $storeId = $storeId->getId();
        }

        $filterType = is_array($storeId) ? 'in' : 'eq';

        parent::addFieldToFilter(
            'store.store_id',
            [
                [$filterType => $storeId],
                ['eq' => '0'],
                ['null' => true]
            ]
        );

        $this->getSelect()->distinct(true);

        return $this;
    }

    /**
     * Join store table
     *
     * @throws LocalizedException
     */
    protected function joinStoreTable()
    {
        $entityInfo = $this->_getAssociatedEntityInfo('store');
        if (!$this->getFlag('is_store_table_joined')) {
            $this->setFlag('is_store_table_joined', true);
            $this->getSelect()->joinLeft(
                ['store' => $this->getTable($entityInfo['associations_table'])],
                'main_table.' . $entityInfo['main_table_id_field'] . ' = store.' .
                $entityInfo['linked_table_id_field'],
                []
            );
            $this->getSelect()->distinct(true);
        }
    }

    /**
     * Retrieve correspondent entity information (associations table name, columns names)
     * of entities associated entity by specified entity type
     *
     * @param string $entityType
     *
     * @return array
     * @throws LocalizedException
     */
    protected function _getAssociatedEntityInfo($entityType)
    {
        if (isset($this->_associatedEntitiesMap[$entityType])) {
            return $this->_associatedEntitiesMap[$entityType];
        }

        throw new LocalizedException(
            __('There is no information about associated entity type "%1".', $entityType)
        );
    }

    /**
     * Left Join table to collection select
     *
     * @param string|array $table
     * @param string $cond
     * @param string|array $cols
     * @return $this
     */
    public function joinLeft($table, $cond, $cols = '*')
    {
        if (is_array($table)) {
            foreach ($table as $k => $v) {
                $alias = $k;
                $table = $v;
                break;
            }
        } else {
            $alias = $table;
        }

        if (!isset($this->_joinedTables[$alias])) {
            $this->getSelect()->joinLeft([$alias => $this->getTable($table)], $cond, $cols);
            $this->_joinedTables[$alias] = true;
        }

        return $this;
    }

    /**
     * Get SQL for get record count
     *
     * @return Select
     * @throws Zend_Db_Select_Exception
     */
    public function getSelectCountSql()
    {
        $this->_renderFilters();

        $countSelect = clone $this->getSelect();
        $countSelect->reset(Select::ORDER);
        $countSelect->reset(Select::LIMIT_COUNT);
        $countSelect->reset(Select::LIMIT_OFFSET);
        $countSelect->reset(Select::COLUMNS);

        if (!count($this->getSelect()->getPart(Select::GROUP))) {
            $countSelect->columns(new Zend_Db_Expr('COUNT(*)'));

            return $countSelect;
        }

        $countSelect->reset(Select::GROUP);
        $group = $this->getSelect()->getPart(Select::GROUP);
        foreach ($group as $groupFieldKey => $groupField) {
            if (!empty($this->_getMappedField($groupField))) {
                $group[$groupFieldKey] = $this->_getMappedField($groupField);
            }
        }
        $countSelect->columns(new Zend_Db_Expr(("COUNT(DISTINCT " . implode(", ", $group) . ")")));

        return $countSelect;
    }

    /**
     * Perform operations after collection load
     *
     * @return \Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection|$this
     * @throws LocalizedException
     */
    protected function _afterLoad()
    {
        $this->joinStoreTable();
        $this->addStoreData();

        return parent::_afterLoad();
    }

    /**
     * Adds store data to the items
     */
    protected function addStoreData()
    {
        $mainTableIdField       = $this->_associatedEntitiesMap['store']['main_table_id_field'];
        $associatedTable        = $this->_associatedEntitiesMap['store']['associations_table'];
        $linkedTableIdFieldName = $this->_associatedEntitiesMap['store']['linked_table_id_field'];
        $storeIdFieldName       = $this->_associatedEntitiesMap['store']['entity_id_field'];

        $ids = $this->getColumnValues($mainTableIdField);
        if (count($ids)) {
            $connection = $this->getConnection();

            $select = $connection->select()->from(
                [
                    $associatedTable => $this->getTable($associatedTable)
                ]
            )->where($associatedTable . '.' . $linkedTableIdFieldName . ' IN (?)', $ids);

            $result = $connection->fetchAll($select);
            $data   = [];
            if ($result) {
                foreach ($result as $storeData) {
                    $data[$storeData[$linkedTableIdFieldName]][] = $storeData[$storeIdFieldName];
                }
            }
            $this->addStoresDataToItems($data);
        }
    }

    /**
     * Add stores to each item
     *
     * @param array $data
     */
    protected function addStoresDataToItems($data)
    {
        $mainTableIdField = $this->_associatedEntitiesMap['store']['main_table_id_field'];

        foreach ($this as $item) {
            $linkedId = $item->getData($mainTableIdField);
            if (!isset($data[$linkedId]) || !$data[$linkedId]) {
                $item->setData('store_ids', [0]);
                continue;
            }

            $storeIdKey = array_search(Store::DEFAULT_STORE_ID, $data[$linkedId], true);
            if ($storeIdKey !== false) {
                $stores    = $this->storeManager->getStores(false, true);
                $storeId   = current($stores)->getId();
                $storeCode = key($stores);
            } else {
                $storeId   = current($data[$linkedId]);
                $store     = $this->storeManager->getStore($storeId);
                $storeCode = $store->getCode();
            }

            $item->setData('_first_store_id', $storeId)
                 ->setData('store_code', $storeCode)
                 ->setData('store_ids', $data[$linkedId]);
        }
    }

    /**
     * Convert items array to array for select options
     *
     * return items array
     * array(
     *      $index => array(
     *          'value' => mixed
     *          'label' => mixed
     *      )
     * )
     *
     * @param string $valueField
     * @param string $labelField
     * @param array $additional
     * @return array
     */
    protected function _toOptionArray($valueField = 'entity_id', $labelField = 'title', $additional = [])
    {
        $res                 = [];
        $additional['value'] = $valueField;
        $additional['label'] = $labelField;

        foreach ($this as $item) {
            foreach ($additional as $code => $field) {
                $data[$code] = $item->getData($field);
            }
            $res[] = $data;
        }

        return $res;
    }

    /**
     * Let do something before add loaded item in collection
     *
     * @param DataObject $item
     * @return DataObject
     */
    protected function beforeAddLoadedItem(DataObject $item)
    {
        if ($item instanceof AbstractModel) {
            $this->_resource->afterLoad($item);
        }

        return parent::beforeAddLoadedItem($item);
    }
}
