<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingCalculatorBase\Model;

use Magento\Customer\Model\Customer;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Quote\Model\Quote;
use Magento\Customer\Model\Address\AbstractAddress as CustomerAddress;
use MageWorx\ShippingCalculatorBase\Api\AddressResolverInterface;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Customer\Model\Session as CustomerSession;
use MageWorx\ShippingCalculatorBase\Helper\Data as Helper;
use Magento\Directory\Model\ResourceModel\Region\CollectionFactory as RegionCollectionFactory;

/**
 * Class GeneralAddressResolver
 *
 * Resolve customers address using quote, session or tax configuration
 */
class GeneralAddressResolver implements AddressResolverInterface
{
    /**
     * @var RegionCollectionFactory
     */
    protected $regionCollectionFactory;

    /**
     * @var CheckoutSession
     */
    protected $checkoutSession;

    /**
     * @var CustomerSession
     */
    protected $customerSession;

    /**
     * @var Helper
     */
    protected $helper;

    /**
     * GeneralAddressResolver constructor.
     *
     * @param RegionCollectionFactory $regionCollectionFactory
     * @param CheckoutSession $checkoutSession
     * @param CustomerSession $customerSession
     * @param Helper $helper
     */
    public function __construct(
        RegionCollectionFactory $regionCollectionFactory,
        CheckoutSession $checkoutSession,
        CustomerSession $customerSession,
        Helper $helper
    ) {
        $this->regionCollectionFactory = $regionCollectionFactory;

        $this->checkoutSession = $checkoutSession;
        $this->customerSession = $customerSession;
        $this->helper          = $helper;
    }

    /**
     * Get default country id
     *
     * @return string
     */
    public function getCountryId(): string
    {
        if ($this->customerSession->getCountryId()) {
            return $this->customerSession->getCountryId();
        }

        $address              = $this->getAddress();
        $countryIdFromAddress = $address ? $address->getCountry() : null;
        if ($countryIdFromAddress) {
            $countryId = $countryIdFromAddress;
        } else {
            $countryId = $this->getDefaultCountryId();
        }

        return $countryId;
    }

    /**
     * Get default region id or code
     *
     * @return string
     */
    public function getRegionId(): string
    {
        if ($this->customerSession->getRegionId()) {
            return $this->customerSession->getRegionId();
        }

        $address = $this->getAddress();
        if ($address instanceof CustomerAddress) {
            $regionIdFromAddress = $address->getRegionId() ?? $address->getRegion();
        } else {
            $regionIdFromAddress = false;
        }

        if ($regionIdFromAddress) {
            $region = $regionIdFromAddress;
        } else {
            $region = $this->getDefaultRegion();
        }

        return $region;
    }

    /**
     * Get default zip (postal) code
     *
     * @return string
     */
    public function getZipCode(): string
    {
        if ($this->customerSession->getPostcode()) {
            return $this->customerSession->getPostcode();
        }

        $address             = $this->getAddress();
        $postcodeFromAddress = $address ? $address->getPostcode() : false;
        if ($postcodeFromAddress) {
            $postcode = $postcodeFromAddress;
        } else {
            $postcode = $this->getDefaultPostcode();
        }

        return $postcode;
    }

    /**
     * Get actual quote
     *
     * @return CartInterface|Quote|null
     */
    protected function getQuote(): ?CartInterface
    {
        return $this->checkoutSession->getQuote();
    }

    /**
     * Get actual address
     *
     * @return CustomerAddress|null
     */
    protected function getAddress(): ?CustomerAddress
    {
        $address = null;

        if ($this->getCustomer()) {
            $address = $this->getCustomer()->getDefaultShippingAddress();
            if ($address && $address->getCountryId()) {
                return $address;
            }
        }

        $quote = $this->getQuote();
        if ($quote) {
            $address = $quote->getShippingAddress();
            if ($address && $address->getCountryId()) {
                return $address;
            }
        }

        return $address;
    }

    /**
     * @return string
     */
    protected function getDefaultCountryId(): string
    {
        $taxCountryId = $this->helper->getTaxDefaultCountryId();
        if ($taxCountryId) {
            return $taxCountryId;
        }

        return 'US';
    }

    /**
     * Returns default region or region id
     *
     * @return string
     */
    protected function getDefaultRegion(): string
    {
        $taxRegion = $this->helper->getTaxDefaultRegion();
        if ($taxRegion && $taxRegion !== '*') {
            return $taxRegion;
        }

        return '';
    }

    /**
     * Get default postcode
     *
     * @return string
     */
    protected function getDefaultPostcode(): string
    {
        $taxPostcode = $this->helper->getTaxDefaultPostcode();
        if ($taxPostcode) {
            return $taxPostcode;
        }

        return '';
    }

    /**
     * Get current customer from session
     *
     * @return Customer|null
     */
    protected function getCustomer(): ?Customer
    {
        return $this->customerSession->getCustomer();
    }
}
