<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingRules\Model\ResourceModel;

use Magento\Framework\App\Cache\TypeListInterface as TypeList;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Framework\Stdlib\StringUtils;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Store\Model\StoreResolver;
use MageWorx\ShippingRules\Cache\Type\ShippingCarriers;
use MageWorx\ShippingRules\Helper\Data as Helper;
use MageWorx\ShippingRules\Model\Carrier as CarrierModel;
use MageWorx\ShippingRules\Model\ResourceModel\Method\CollectionFactory;

class Carrier extends AbstractResourceModel
{
    /**
     * Store associated with carrier entities information map
     *
     * @var array
     */
    protected $_associatedEntitiesMap = [
        'store' => [
            'associations_table' => CarrierModel::CARRIER_TABLE_NAME . '_store',
            'ref_id_field'       => 'entity_id',
            'entity_id_field'    => 'store_id',
        ]
    ];

    /**
     * @var Method\CollectionFactory
     */
    protected $methodsCollectionFactory;

    /**
     * @var StoreResolver
     */
    protected $storeResolver;

    /**
     * @var TypeList
     */
    protected $typeList;

    /**
     * @param Context $context
     * @param StringUtils $string
     * @param Helper $helper
     * @param StoreManagerInterface $storeManager
     * @param StoreResolver $storeResolver
     * @param CollectionFactory $methodsCollectionFactory
     * @param TypeList|null $typeList
     * @param string|null $connectionName
     */
    public function __construct(
        Context               $context,
        StringUtils           $string,
        Helper                $helper,
        StoreManagerInterface $storeManager,
        StoreResolver         $storeResolver,
        CollectionFactory     $methodsCollectionFactory,
        TypeList              $typeList,
        ?string               $connectionName = null
    ) {
        $this->methodsCollectionFactory = $methodsCollectionFactory;
        $this->storeResolver            = $storeResolver;
        $this->typeList                 = $typeList;
        parent::__construct($context, $string, $helper, $storeManager, $connectionName);
    }

    /**
     * @param AbstractModel $object
     * @return $this
     * @throws LocalizedException
     */
    public function _beforeSave(AbstractModel $object)
    {
        parent::_beforeSave($object);
        $this->validateModel($object);

        return $this;
    }

    /**
     * Validate model required fields
     *
     * @param AbstractModel $object
     * @throws LocalizedException
     */
    public function validateModel(AbstractModel $object)
    {
        /** @var Carrier $object */
        if (!$object->getCarrierCode()) {
            throw new LocalizedException(__('Carrier Code is required'));
        }
    }

    /**
     * @param AbstractModel $object
     * @return Carrier
     */
    protected function _afterSave(AbstractModel $object)
    {
        parent::_afterSave($object);
        $this->invalidateCarriersCache();

        return $this;
    }

    /**
     * Invalidate related cache
     */
    protected function invalidateCarriersCache()
    {
        $this->typeList->invalidate(ShippingCarriers::TYPE_IDENTIFIER);
    }

    /**
     * Initialize main table and table id field
     *
     * @return void
     */
    protected function _construct()
    {
        $this->_init(CarrierModel::CARRIER_TABLE_NAME, 'carrier_id');
    }

    /**
     * Add customer group ids and store ids to rule data after load
     *
     * @param AbstractModel $object
     * @return $this
     */
    protected function _afterLoad(AbstractModel $object)
    {
        /** @var CarrierModel $object */
        parent::_afterLoad($object);

        return $this;
    }

    /**
     * Get store labels table
     *
     * @return string
     */
    protected function getStoreLabelsTable()
    {
        return $this->getTable(CarrierModel::CARRIER_LABELS_TABLE_NAME);
    }

    /**
     * Get reference id column name from the labels table
     *
     * @return string
     */
    protected function getStoreLabelsTableRefId()
    {
        return 'carrier_id';
    }
}
