<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */
declare(strict_types = 1);

namespace MageWorx\ShippingRules\Api\Data;

use Magento\Framework\Api\CustomAttributesDataInterface;

/**
 * Interface MethodInterface
 */
interface MethodInterface extends CustomAttributesDataInterface
{
    public const EDT_DISPLAY_TYPE_DAYS           = 0;
    public const EDT_DISPLAY_TYPE_HOURS          = 1;
    public const EDT_DISPLAY_TYPE_DAYS_AND_HOURS = 2;

    public const EDT_PLACEHOLDER_MIN = '{{min}}';
    public const EDT_PLACEHOLDER_MAX = '{{max}}';

    public const ENTITY_ID_FIELD_NAME = 'entity_id';

    /**
     * Retrieve method title
     *
     * @return string
     */
    public function getTitle(): string;

    /**
     * Flag: is the title replacement allowed
     * In case it is allowed - the title from a most prior rate will be used
     * (in a case valid rate is exists)
     *
     * @return bool
     */
    public function getReplaceableTitle(): bool;

    /**
     * Retrieve method code
     *
     * @return string
     */
    public function getCode(): string;

    /**
     * Retrieve method ID
     *
     * @return int
     */
    public function getEntityId();

    /**
     * Check is method active
     *
     * @return bool
     */
    public function getActive(): bool;

    /**
     * Default method price
     *
     * @return float (12,2)
     */
    public function getPrice(): float;

    /**
     * Get Max price threshold
     *
     * @return float
     */
    public function getMaxPriceThreshold(): float;

    /**
     * Get Min price threshold
     *
     * @return float
     */
    public function getMinPriceThreshold(): float;

    /**
     * Flag: is need to ignore price of the virtual product in cart during rate validation
     *
     * @return bool
     */
    public function getIgnoreVirtualProductsPrice(): bool;

    /**
     * Flag: is price condition include discount
     *
     * @return bool
     */
    public function getUsePriceWithDiscount(): bool;

    /**
     * Default method cost
     *
     * @return float (12,2)
     */
    public function getCost(): float;

    /**
     * Get created at date
     *
     * @return mixed
     */
    public function getCreatedAt();

    /**
     * Get last updated date
     *
     * @return mixed
     */
    public function getUpdatedAt();

    /**
     * Check is we should disable this method when there are no valid rates
     *
     * @return bool
     */
    public function getDisabledWithoutValidRates(): bool;

    /**
     * Multiple rates price calculation method
     *
     * @return int
     * @see \MageWorx\ShippingRules\Model\Config\Source\MultipleRatesPrice::toOptionArray()
     *
     */
    public function getMultipleRatesPrice(): int;

    /**
     * Is free shipping by a third party extension allowed (like sales rule)
     *
     * @return bool
     */
    public function getAllowFreeShipping(): bool;

    /**
     * Min estimated delivery time (can be overwritten by a value form a rate, visible at checkout & cart)
     *
     * @return float
     */
    public function getEstimatedDeliveryTimeMin(): float;

    /**
     * Max estimated delivery time (can be overwritten by a value form a rate, visible at checkout & cart)
     *
     * @return float
     */
    public function getEstimatedDeliveryTimeMax(): float;

    /**
     * Flag: is replacing of the estimated delivery time allowed (from a valid rates)
     *
     * @return bool
     */
    public function getReplaceableEstimatedDeliveryTime(): bool;

    /**
     * How an estimated delivery time values would be visible for the customer?
     *
     * Possible values:
     * DAYS (rounded) - MethodInterface::EDT_DISPLAY_TYPE_DAYS
     * HOURS - MethodInterface::EDT_DISPLAY_TYPE_HOURS
     * DAYS & HOURS - MethodInterface::EDT_DISPLAY_TYPE_DAYS_AND_HOURS
     *
     * @return int
     */
    public function getEstimatedDeliveryTimeDisplayType(): int;

    /**
     * Flag: should be the Estimated Delivery Time displayed for the customer or not
     *
     * @return bool
     */
    public function getShowEstimatedDeliveryTime(): bool;

    /**
     * Markup for the EDT message.
     * You can use variables {{min}} {{max}} which will be replaced by a script to the corresponding values
     * from a method or rate.
     *
     * {{min}} - MethodInterface::EDT_PLACEHOLDER_MIN
     * {{max}} - MethodInterface::EDT_PLACEHOLDER_MAX
     *
     * @return string
     */
    public function getEstimatedDeliveryTimeMessage(): string;

    /**
     * Get associated store Ids
     *
     * @return int[]
     */
    public function getStoreIds(): array;

    /**
     * Get corresponding carrier code (relation)
     *
     * @return string
     */
    public function getCarrierCode(): string;

    /**
     * Get method store specific labels
     *
     * @return string[]
     */
    public function getStoreLabels(): array;

    /**
     * Set if not yet and retrieve method store specific EDT messages
     *
     * @return string[]
     */
    public function getEdtStoreSpecificMessages(): array;

    //_______________________________________________SETTERS___________________________________________________________

    /**
     * Set method title
     *
     * @param string $value
     * @return MethodInterface
     */
    public function setTitle(string $value): MethodInterface;

    /**
     * Flag: is the title replacement allowed
     * In case it is allowed - the title from a most prior rate will be used
     * (in a case valid rate is exists)
     *
     * @param bool $value
     * @return MethodInterface
     */
    public function setReplaceableTitle(bool $value): MethodInterface;

    /**
     * Set method code
     *
     * @param string $value
     * @return MethodInterface
     */
    public function setCode(string $value): MethodInterface;

    /**
     * Set corresponding carrier id
     *
     * @param int $id
     * @return MethodInterface
     */
    public function setCarrierId(int $id): MethodInterface;

    /**
     * Set method ID
     *
     * @param int $value
     * @return MethodInterface
     */
    public function setEntityId($value);

    /**
     * Set is method active
     *
     * @param bool $value
     * @return MethodInterface
     */
    public function setActive(bool $value): MethodInterface;

    /**
     * Set default method price
     *
     * @param float $value
     * @return MethodInterface
     */
    public function setPrice(float $value): MethodInterface;

    /**
     * Set Max price threshold
     *
     * @param float $value
     * @return MethodInterface
     */
    public function setMaxPriceThreshold(float $value): MethodInterface;

    /**
     * Set Min price threshold
     *
     * @param float $value
     * @return MethodInterface
     */
    public function setMinPriceThreshold(float $value): MethodInterface;

    /**
     * Flag: is need to ignore price of the virtual product in cart during rate validation
     *
     * @param bool $value
     * @return MethodInterface
     */
    public function setIgnoreVirtualProductsPrice(bool $value): MethodInterface;

    /**
     * Flag: is price condition include discount
     *
     * @param bool $value
     * @return MethodInterface
     */
    public function setUsePriceWithDiscount(bool $value): MethodInterface;

    /**
     * Set Default method cost
     *
     * @param float $value
     * @return MethodInterface
     */
    public function setCost(float $value): MethodInterface;

    /**
     * Set created at date
     *
     * @param mixed $value
     * @return MethodInterface
     */
    public function setCreatedAt($value): MethodInterface;

    /**
     * Set last updated date
     *
     * @param mixed $value
     * @return MethodInterface
     */
    public function setUpdatedAt($value): MethodInterface;

    /**
     * Set is we should disable this method when there are no valid rates
     *
     * @param bool $value
     * @return MethodInterface
     */
    public function setDisabledWithoutValidRates(bool $value): MethodInterface;

    /**
     * Set Multiple rates price calculation method
     *
     * @param int $value
     * @return MethodInterface
     * @see \MageWorx\ShippingRules\Model\Config\Source\MultipleRatesPrice::toOptionArray($value)
     *
     */
    public function setMultipleRatesPrice(int $value): MethodInterface;

    /**
     * Is free shipping by a third party extension allowed (like sales rule)
     *
     * @param bool $value
     * @return MethodInterface
     */
    public function setAllowFreeShipping(bool $value): MethodInterface;

    /**
     * Set Min estimated delivery time (can be overwritten by a value form a rate, visible at checkout & cart)
     *
     * @param float $value
     * @return MethodInterface
     */
    public function setEstimatedDeliveryTimeMin(float $value): MethodInterface;

    /**
     * Set Max estimated delivery time (can be overwritten by a value form a rate, visible at checkout & cart)
     *
     * @param float $value
     * @return MethodInterface
     */
    public function setEstimatedDeliveryTimeMax(float $value): MethodInterface;

    /**
     * Set Flag: is replacing of the estimated delivery time allowed (from a valid rates)
     *
     * @param bool $value
     * @return MethodInterface
     */
    public function setReplaceableEstimatedDeliveryTime(bool $value): MethodInterface;

    /**
     * How an estimated delivery time values would be visible for the customer?
     *
     * Possible values:
     * DAYS (rounded) - MethodInterface::EDT_DISPLAY_TYPE_DAYS
     * HOURS - MethodInterface::EDT_DISPLAY_TYPE_HOURS
     * DAYS & HOURS - MethodInterface::EDT_DISPLAY_TYPE_DAYS_AND_HOURS
     *
     * @param int $value
     * @return MethodInterface
     */
    public function setEstimatedDeliveryTimeDisplayType(int $value): MethodInterface;

    /**
     * Set Flag: should be the Estimated Delivery Time displayed for the customer or not
     *
     * @param bool $value
     * @return MethodInterface
     */
    public function setShowEstimatedDeliveryTime(bool $value): MethodInterface;

    /**
     * Set estimated Delivery Time Message
     *
     * Markup for the EDT message.
     * You can use variables {{min}} {{max}} which will be replaced by a script to the corresponding values
     * from a method or rate.
     *
     * {{min}} - MethodInterface::EDT_PLACEHOLDER_MIN
     * {{max}} - MethodInterface::EDT_PLACEHOLDER_MAX
     *
     * @param string|null $value
     * @return MethodInterface
     */
    public function setEstimatedDeliveryTimeMessage(?string $value): MethodInterface;

    /**
     * Set associated store Ids
     *
     * @param int[] $value
     * @return MethodInterface
     */
    public function setStoreIds(array $value): MethodInterface;

    /**
     * Set corresponding carrier code
     *
     * @param string $code
     * @return MethodInterface
     */
    public function setCarrierCode(string $code): MethodInterface;

    /**
     * Set store specific labels (title)
     *
     * @param string[] $storeLabels
     * @return MethodInterface
     */
    public function setStoreLabels(array $storeLabels = []): MethodInterface;

    /**
     * Set Store Specific Estimated Delivery Time Messages
     *
     * @param string[] $messages
     * @return MethodInterface
     */
    public function setEdtStoreSpecificMessages(array $messages = []): MethodInterface;
}
