<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingRules\Model\Plugin\Shipping\Rate\Result;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Request\Http;
use Magento\Framework\Phrase;
use Magento\Quote\Model\Quote\Address\RateResult\Error;
use Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory;
use Magento\Quote\Model\Quote\Address\RateResult\Method;
use Magento\Shipping\Model\Rate\Result;
use MageWorx\ShippingRules\Helper\Data;
use MageWorx\ShippingRules\Model\Plugin\CollectValidMethods;
use MageWorx\ShippingRules\Model\Rule;
use MageWorx\ShippingRules\Model\RulesApplier;
use function in_array;

class GetAllRates
{
    /**
     * @var ErrorFactory
     */
    protected $errorFactory;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var Http
     */
    private $request;

    /**
     * @var CollectValidMethods
     */
    private $collectValidMethodsPlugin;

    /**
     * @var RulesApplier
     */
    private $rulesApplier;

    /**
     * GetAllRates constructor.
     *
     * @param ErrorFactory $errorFactory
     * @param ScopeConfigInterface $scopeConfig
     * @param Data $helper
     * @param Http $request
     * @param CollectValidMethods $collectValidMethodsPlugin
     * @param RulesApplier $rulesApplier
     */
    public function __construct(
        ErrorFactory         $errorFactory,
        ScopeConfigInterface $scopeConfig,
        Data                 $helper,
        Http                 $request,
        CollectValidMethods  $collectValidMethodsPlugin,
        RulesApplier         $rulesApplier
    ) {
        $this->errorFactory              = $errorFactory;
        $this->scopeConfig               = $scopeConfig;
        $this->helper                    = $helper;
        $this->request                   = $request;
        $this->collectValidMethodsPlugin = $collectValidMethodsPlugin;
        $this->rulesApplier              = $rulesApplier;
    }

    /**
     * Disable the marked shipping rates. Rates disabling in the
     *
     * @param Result $subject
     * @param array $result
     * @return array
     * @see RulesApplier::disableShippingMethod
     *
     * NOTE: If you can not see some of the shipping rates, start debugging from here. At first, check 'is_disabled'
     * param in the shipping rate object.
     *
     */
    public function afterGetAllRates($subject, $result)
    {
        if ($this->request->getRouteName() == 'multishipping') {
            /**
             * This plugin should work only on the regular checkout/cart
             */
            return $result;
        }

        if ($this->helper->getShippingPerProduct()) {
            $availableShippingMethods = $this->collectValidMethodsPlugin->getAvailableShippingMethods();
        } else {
            $availableShippingMethods = [];
        }

        $appliedCarriers = [];

        /**
         * Filter rates in result by minimal price to display only one rate from stack
         */
        $this->filterRatesByMinPrice($result);

        /** @var Method[] $result */
        /**
         * @var int $key
         * @var Method $rate
         */
        foreach ($result as $key => $rate) {
            $code = Rule::getMethodCode($rate);
            $code = trim(strtolower($code));

            if ($this->helper->getShippingPerProduct()) {
                $rateIsAvailable = in_array($code, $availableShippingMethods);
            } else {
                $rateIsAvailable = true;
            }

            if ($rate->getIsDisabled() || !$rateIsAvailable) {
                if ($rate->getShowError() === true) {
                    if (!empty($appliedCarriers[$rate->getCarrier()])) {
                        unset($result[$key]);
                        continue;
                    } else {
                        $appliedCarriers[$rate->getCarrier()] = 1;
                    }

                    if ($rate instanceof Error) {
                        continue;
                    }

                    /** @var Error $error */
                    $error = $this->errorFactory->create();
                    $error->setCarrier($rate->getCarrier());
                    $error->setMethod($rate->getMethod());
                    $error->setCarrierTitle($rate->getCarrierTitle());
                    $defaultErrorMessage = $this->getDefaultErrorMessage($rate->getCarrier());
                    $customErrorMessage  = $rate->getCustomErrorMessage();
                    $error->setErrorMessage($customErrorMessage ? $customErrorMessage : $defaultErrorMessage);
                    $result[$key] = $error;
                } elseif ($rate->getShowError() === false) {
                    unset($result[$key]);
                } else {
                    // skip the rate state as is
                }
            }
        }

        if ($this->helper->isNeedToSortCarriers()) {
            uasort(
                $result,
                function ($first, $second) {
                    return $first->getCarrierSortOrder() > $second->getCarrierSortOrder() ? -1 : 1;
                }
            );
        }

        if ($this->helper->displayCheapestRateAtTop()) {
            uasort(
                $result,
                function ($first, $second) {
                    return $first->getPrice() < $second->getPrice() ? -1 : 1;
                }
            );
        }

        return $result;
    }

    /**
     * Filter rates by minimal price
     *
     * @param array $result
     * @return array
     */
    private function filterRatesByMinPrice(array $result = [])
    {
        if (count($result) <= 1) {
            return $result;
        }

        $rulesWithFilteredRates = $this->rulesApplier->getShippingMethodsFilterRules();
        ksort($rulesWithFilteredRates, SORT_NUMERIC);

        /** @var Rule[] $rules */
        $rules = [];
        foreach ($rulesWithFilteredRates as $sortedRules) {
            $rules += $sortedRules;
        }

        foreach ($rules as $rule) {
            $result = $rule->filterRatesByMinimalPrice($result);
        }

        return $result;
    }

    /**
     * @param string $carrierCode
     * @return Phrase|string
     */
    private function getDefaultErrorMessage($carrierCode)
    {
        return $this->scopeConfig->getValue('carriers/' . $carrierCode . '/specificerrmsg') ?
            $this->scopeConfig->getValue('carriers/' . $carrierCode . '/specificerrmsg') :
            __('Sorry, but we can\'t deliver to the destination country with this shipping module.');
    }
}
