<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingRules\Model;

use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use MageWorx\ShippingRules\Api\Data\ExtendedZoneDataInterface;
use MageWorx\ShippingRules\Api\ExtendedZoneInterface;
use MageWorx\ShippingRules\Helper\Image;

class ExtendedZone extends AbstractModel implements ExtendedZoneInterface, ExtendedZoneDataInterface
{
    public const EXTENDED_ZONE_TABLE_NAME        = 'mageworx_shippingrules_extended_zone';
    public const EXTENDED_ZONE_STORE_TABLE_NAME  = 'mageworx_shippingrules_extended_zone_store';
    public const EXTENDED_ZONE_LABELS_TABLE_NAME = 'mageworx_shippingrules_extended_zone_labels';

    public const REGISTRY_KEY = 'extended_zone';

    /**
     * Prefix of model events names
     *
     * @var string
     */
    protected $_eventPrefix = 'mageworx_shippingrules_extended_zone';

    /**
     * Parameter name in event
     *
     * In observe method you can use $observer->getEvent()->getExtendedZone() in this case
     *
     * @var string
     */
    protected $_eventObject = 'extended_zone';

    /**
     * @var Image
     */
    protected $helper;

    /**
     * @var array
     */
    protected $storeLabels;

    /**
     * @param Context $context
     * @param Registry $registry
     * @param AbstractResource|null $resource
     * @param AbstractDb|null $resourceCollection
     * @param array $data
     */
    public function __construct(
        Context           $context,
        Registry          $registry,
        Image             $helper,
        ?AbstractResource $resource = null,
        ?AbstractDb       $resourceCollection = null,
        array             $data = []
    ) {
        parent::__construct(
            $context,
            $registry,
            $resource,
            $resourceCollection,
            $data
        );
        $this->helper = $helper;
    }

    /**
     * Get url to the image, if exist
     *
     * @return string
     */
    public function getImageUrl()
    {
        $imagePath = $this->getImage();
        if (!$imagePath) {
            return '';
        }

        return $this->helper->getMediaUrl($imagePath);
    }

    /**
     * Image path (relative). Used as zone preview on frontend
     *
     * @return string
     */
    public function getImage()
    {
        return $this->getData('image');
    }

    /**
     * Return priority of the current zone (sort order)
     *
     * @return int
     */
    public function getPriority()
    {
        return (int)$this->getData('priority');
    }

    /**
     * @return boolean
     */
    public function getIsActive()
    {
        return (bool)$this->getData('is_active');
    }

    /**
     * Get label for store
     *
     * @param null $storeId
     * @return string
     */
    public function getLabel($storeId = null)
    {
        $labels = $this->getStoreLabels();
        if ($storeId != null && !empty($labels[$storeId])) {
            return $labels[$storeId];
        }

        return $this->getName();
    }

    /**
     * Get corresponding store labels
     * where the key is store view id (int), value is label (string)
     *
     * @return array
     */
    public function getStoreLabels()
    {
        if (empty($this->storeLabels) && !empty($this->getData('store_labels'))) {
            $this->storeLabels = $this->getData('store_labels');
        } elseif (empty($this->storeLabels)) {
            $this->storeLabels = $this->getResource()->getStoreLabels($this->getId());
        }

        return $this->storeLabels;
    }

    /**
     * Unique zone name
     *
     * @return string
     */
    public function getName()
    {
        return $this->getData('name');
    }

    /**
     * Zone description text
     *
     * @return string
     */
    public function getDescription()
    {
        return $this->getData('description');
    }

    /**
     * List of countries assigned to the zone
     *
     * @return array
     */
    public function getCountriesId()
    {
        $countries = $this->getData('countries_id');
        if (is_string($countries)) {
            $countries = explode(',', $countries);
            $this->setData('countries_id', $countries);
        }

        return $countries;
    }

    /**
     * @return string
     */
    public function getCreatedAt()
    {
        return $this->getData('created_at');
    }

    /**
     * @return string
     */
    public function getUpdatedAt()
    {
        return $this->getData('updated_at');
    }

    /**
     * Set resource model and Id field name
     *
     * @return void
     */
    protected function _construct()
    {
        parent::_construct();
        $this->_init('MageWorx\ShippingRules\Model\ResourceModel\ExtendedZone');
        $this->setIdFieldName('entity_id');
    }
}
