<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingCalculatorBase\Test\Unit\Model;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Shipping\Model\Shipping;
use MageWorx\ShippingCalculatorBase\Helper\Data as Helper;
use MageWorx\ShippingCalculatorBase\Model\EstimateShippingMethods;
use MageWorx\ShippingCalculatorBase\Plugin\ShippingRules\QuoteSubstitution as QuoteSubstitutionPlugin;

class EstimateShippingMethodsTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var object|EstimateShippingMethods
     */
    private $estimateClass;

    /**
     * @var RateRequestFactory
     */
    private $rateRequestFactoryMock;

    /**
     * @var Shipping
     */
    private $shippingMock;

    /**
     * @var CartInterfaceFactory
     */
    private $cartFactoryMock;

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepositoryMock;

    /**
     * @var DataObjectFactory
     */
    private $dataObjectFactoryMock;

    /**
     * @var CartInterface
     */
    private $cartMock;

    /**
     * @var Helper
     */
    private $helperMock;

    /**
     * @var QuoteSubstitutionPlugin
     */
    private $quoteSubstitutionPluginMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    private $rateRequestMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    private $shippingAddressMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    private $productMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    private $rateResultMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    private $storeManagerMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    private $storeMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    private $currencyMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    private $currencyFilterMock;

    /**
     * @var array
     */
    private $rates;

    /**
     * @inheritdoc
     */
    public function setUp(): void
    {
        $objectManager = new \Magento\Framework\TestFramework\Unit\Helper\ObjectManager($this);

        // Mock
        $this->rateRequestMock = $this->getMockBuilder(
            \Magento\Quote\Model\Quote\Address\RateRequest::class
        )
                                      ->disableOriginalConstructor()
                                      ->getMock();

        $this->rateRequestFactoryMock = $this->getMockBuilder(
            \Magento\Quote\Model\Quote\Address\RateRequestFactory::class
        )
                                             ->disableOriginalConstructor()
                                             ->setMethods(['create'])
                                             ->getMock();

        $this->shippingMock = $this->getMockBuilder(
            \Magento\Shipping\Model\Shipping::class
        )
                                   ->disableOriginalConstructor()
                                   ->getMock();

        $this->cartMock = $this->getMockBuilder(
            \Magento\Quote\Model\Quote::class
        )
                               ->disableOriginalConstructor()
                               ->getMock();

        $this->cartFactoryMock = $this->getMockBuilder(
            \Magento\Quote\Api\Data\CartInterfaceFactory::class
        )
                                      ->disableOriginalConstructor()
                                      ->getMock();

        $this->shippingAddressMock = $this->getMockBuilder(
            \Magento\Quote\Model\Quote\Address::class
        )
                                          ->disableOriginalConstructor()
                                          ->getMock();

        $this->productRepositoryMock = $this->getMockBuilder(
            \Magento\Catalog\Api\ProductRepositoryInterface::class
        )
                                            ->disableOriginalConstructor()
                                            ->getMock();

        $this->productMock = $this->getMockBuilder(
            \Magento\Catalog\Model\Product::class
        )
                                  ->disableOriginalConstructor()
                                  ->getMock();

        $this->rateResultMock = $this->getMockBuilder(
            \Magento\Shipping\Model\Rate\Result::class
        )
                                     ->disableOriginalConstructor()
                                     ->getMock();

        $this->dataObjectFactoryMock = $this->getMockBuilder(
            \Magento\Framework\DataObjectFactory::class
        )
                                            ->disableOriginalConstructor()
                                            ->getMock();

        $this->helperMock = $this->getMockBuilder(
            \MageWorx\ShippingCalculatorBase\Helper\Data::class
        )
                                 ->disableOriginalConstructor()
                                 ->getMock();

        $this->quoteSubstitutionPluginMock = $this->getMockBuilder(
            \MageWorx\ShippingCalculatorBase\Plugin\ShippingRules\QuoteSubstitution::class
        )
                                                  ->disableOriginalConstructor()
                                                  ->getMock();

        $this->storeManagerMock = $this->getMockBuilder(
            \Magento\Store\Model\StoreManagerInterface::class
        )
                                       ->disableOriginalConstructor()
                                       ->getMock();

        $this->storeMock = $this->getMockBuilder(
            \Magento\Store\Model\Store::class
        )
                                ->disableOriginalConstructor()
                                ->getMock();

        $this->currencyMock = $this->getMockBuilder(
            \Magento\Directory\Model\Currency::class
        )
                                   ->disableOriginalConstructor()
                                   ->getMock();

        $this->currencyFilterMock = $this->getMockBuilder(
            \Magento\Directory\Model\Currency\Filter::class
        )
                                         ->disableOriginalConstructor()
                                         ->getMock();

        $priceCurrency = $this->getMockBuilder(PriceCurrencyInterface::class)->getMock();
        $priceCurrency->expects($this->any())->method('round')->willReturnArgument(0);

        $this->rates = [
            $objectManager->getObject(
                \Magento\Quote\Model\Quote\Address\RateResult\Method::class,
                [
                    'priceCurrency' => $priceCurrency,
                    'data'          => [
                        'carrier'     => 'testcarriercode',
                        'method'      => 'testmethodcode',
                        'price'       => '0.99',
                        'methodTitle' => 'Test Title In',
                    ]
                ]
            ),
            $objectManager->getObject(
                \Magento\Quote\Model\Quote\Address\RateResult\Method::class,
                [
                    'priceCurrency' => $priceCurrency,
                    'data'          => [
                        'carrier'     => 'testcarriercode',
                        'method'      => 'test2methodcode',
                        'price'       => '100',
                        'methodTitle' => 'Test Title In 2',
                    ]
                ]
            ),
        ];

        // Create base object
        $this->estimateClass = $objectManager->getObject(
            EstimateShippingMethods::class,
            [
                'rateRequestFactory'      => $this->rateRequestFactoryMock,
                'shipping'                => $this->shippingMock,
                'cartFactory'             => $this->cartFactoryMock,
                'productRepository'       => $this->productRepositoryMock,
                'dataObjectFactory'       => $this->dataObjectFactoryMock,
                'helper'                  => $this->helperMock,
                'quoteSubstitutionPlugin' => $this->quoteSubstitutionPluginMock,
                'storeManager'            => $this->storeManagerMock
            ]
        );
    }

    /**
     * Setup store mock
     */
    private function setupStoreMock()
    {
        $this->storeMock->expects($this->atLeastOnce())->method('getId')->willReturn(1);
        $this->storeMock->expects($this->atLeastOnce())->method('getWebsiteId')->willReturn(1);
        $this->storeMock->expects($this->atLeastOnce())->method('getBaseCurrency')->willReturn($this->currencyMock);
        $this->storeMock->expects($this->atLeastOnce())->method('getCurrentCurrency')->willReturn($this->currencyMock);
        $this->currencyMock->expects($this->any())->method('getFilter')->willReturn($this->currencyFilterMock);

        $this->storeManagerMock->expects($this->atLeastOnce())->method('getStore')->willReturn($this->storeMock);
    }

    /**
     * Estimate without product id in request will throw exception
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function testEstimateWithoutProductId()
    {
        $this->cartFactoryMock->expects($this->atLeastOnce())->method('create')->willReturn($this->cartMock);
        $this->rateRequestFactoryMock->expects($this->atLeastOnce())->method('create')->willReturn(
            $this->rateRequestMock
        );
        $this->cartMock->expects($this->atLeastOnce())->method('getShippingAddress')->willReturn(
            $this->shippingAddressMock
        );

        $this->setUpStoreMock();

        $this->expectExceptionMessage((string)__('Product id is missed.'));
        $this->estimateClass->estimate([], []);
    }

    /**
     * Estimate with incomplete data will return empty result
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function testEstimateWithProductId()
    {
        $this->cartFactoryMock->expects($this->atLeastOnce())->method('create')->willReturn($this->cartMock);
        $this->rateRequestFactoryMock->expects($this->atLeastOnce())->method('create')->willReturn(
            $this->rateRequestMock
        );
        $this->cartMock->expects($this->atLeastOnce())->method('getShippingAddress')->willReturn(
            $this->shippingAddressMock
        );
        $this->productRepositoryMock->expects($this->atLeastOnce())->method('getById')->willReturn($this->productMock);
        $this->shippingMock->expects($this->atLeastOnce())->method('getResult')->willReturn($this->rateResultMock);
        $this->rateResultMock->expects($this->atLeastOnce())->method('sortRatesByPrice')->willReturnSelf();
        $this->rateResultMock->expects($this->atLeastOnce())->method('getAllRates')->willReturn([]);

        $this->setUpStoreMock();

        $this->assertEquals([], $this->estimateClass->estimate([], ['product' => 1]));
    }

    /**
     * When rate exists it should be returned (not empty response)
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function testDataOutut()
    {
        $this->cartFactoryMock->expects($this->atLeastOnce())->method('create')->willReturn($this->cartMock);
        $this->rateRequestFactoryMock->expects($this->atLeastOnce())->method('create')->willReturn(
            $this->rateRequestMock
        );
        $this->cartMock->expects($this->atLeastOnce())->method('getShippingAddress')->willReturn(
            $this->shippingAddressMock
        );
        $this->productRepositoryMock->expects($this->atLeastOnce())->method('getById')->willReturn($this->productMock);
        $this->shippingMock->expects($this->atLeastOnce())->method('getResult')->willReturn($this->rateResultMock);
        $this->rateResultMock->expects($this->atLeastOnce())->method('sortRatesByPrice')->willReturnSelf();

        $this->setUpStoreMock();

        $rates = $this->rates;
        $this->rateResultMock->expects($this->atLeastOnce())->method('getAllRates')->willReturn($rates);

        $this->assertNotEmpty($this->estimateClass->estimate([], ['product' => 1]));
    }

    /**
     * Data from config will overwrite existing data (for carrier)
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function testDataRewriteByCarrier()
    {
        $this->cartFactoryMock->expects($this->atLeastOnce())->method('create')->willReturn($this->cartMock);
        $this->rateRequestFactoryMock->expects($this->atLeastOnce())->method('create')->willReturn(
            $this->rateRequestMock
        );
        $this->cartMock->expects($this->atLeastOnce())->method('getShippingAddress')->willReturn(
            $this->shippingAddressMock
        );
        $this->productRepositoryMock->expects($this->atLeastOnce())->method('getById')->willReturn($this->productMock);
        $this->shippingMock->expects($this->atLeastOnce())->method('getResult')->willReturn($this->rateResultMock);
        $this->rateResultMock->expects($this->atLeastOnce())->method('sortRatesByPrice')->willReturnSelf();

        $rates = $this->rates;
        $this->rateResultMock->expects($this->atLeastOnce())->method('getAllRates')->willReturn($rates);

        $descriptionForTest      = 'Not a long description';
        $configFromModuleBackend = [
            'testcarriercode' => [
                'description' => $descriptionForTest
            ]
        ];
        $this->helperMock->expects($this->atLeastOnce())->method('getShippingMethodsConfiguration')->willReturn(
            $configFromModuleBackend
        );

        $this->setUpStoreMock();

        // Execute
        $result = $this->estimateClass->estimate([], ['product' => 1]);

        // Validate
        $this->assertNotEmpty($result);
        $this->assertArrayHasKey('0', $result);
        $this->assertArrayHasKey('1', $result);
        $this->assertNotEmpty($result[0]);
        $this->assertNotEmpty($result[1]);

        $dataMethod1 = $result[0];
        $this->assertArrayHasKey('code', $dataMethod1);
        $this->assertEquals('testcarriercode_testmethodcode', $dataMethod1['code']);
        $this->assertArrayHasKey('description', $dataMethod1);
        $this->assertNotEmpty($dataMethod1['description']);
        $this->assertEquals($dataMethod1['description'], $descriptionForTest);

        $dataMethod2 = $result[1];
        $this->assertArrayHasKey('code', $dataMethod2);
        $this->assertEquals('testcarriercode_test2methodcode', $dataMethod2['code']);
        $this->assertArrayHasKey('description', $dataMethod2);
        $this->assertNotEmpty($dataMethod2['description']);
        $this->assertEquals($dataMethod2['description'], $descriptionForTest);

        $this->assertEquals($dataMethod1['description'], $dataMethod2['description']);
    }

    /**
     * Data from config will overwrite existing data (for carrier) and it will be overwriten by method configuration
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function testDataRewriteByMethodOverCarrier()
    {
        $this->cartFactoryMock->expects($this->atLeastOnce())->method('create')->willReturn($this->cartMock);
        $this->rateRequestFactoryMock->expects($this->atLeastOnce())->method('create')->willReturn(
            $this->rateRequestMock
        );
        $this->cartMock->expects($this->atLeastOnce())->method('getShippingAddress')->willReturn(
            $this->shippingAddressMock
        );
        $this->productRepositoryMock->expects($this->atLeastOnce())->method('getById')->willReturn($this->productMock);
        $this->shippingMock->expects($this->atLeastOnce())->method('getResult')->willReturn($this->rateResultMock);
        $this->rateResultMock->expects($this->atLeastOnce())->method('sortRatesByPrice')->willReturnSelf();

        $rates = $this->rates;
        $this->rateResultMock->expects($this->atLeastOnce())->method('getAllRates')->willReturn($rates);

        $carrierDescriptionForTest = 'Not a long description';
        $methodDescriptionForTest  = 'Another Description will overwrite carrier description';
        $configFromModuleBackend   = [
            'testcarriercode'                 => [
                'description' => $carrierDescriptionForTest
            ],
            'testcarriercode_test2methodcode' => [
                'description' => $methodDescriptionForTest
            ]
        ];
        $this->helperMock->expects($this->atLeastOnce())->method('getShippingMethodsConfiguration')->willReturn(
            $configFromModuleBackend
        );

        $this->setUpStoreMock();

        // Execute
        $result = $this->estimateClass->estimate([], ['product' => 1]);

        // Validate
        $this->assertNotEmpty($result);
        $this->assertArrayHasKey('0', $result);
        $this->assertArrayHasKey('1', $result);
        $this->assertNotEmpty($result[0]);
        $this->assertNotEmpty($result[1]);

        $dataMethod1 = $result[0];
        $this->assertArrayHasKey('code', $dataMethod1);
        $this->assertEquals('testcarriercode_testmethodcode', $dataMethod1['code']);
        $this->assertArrayHasKey('description', $dataMethod1);
        $this->assertNotEmpty($dataMethod1['description']);
        $this->assertEquals($dataMethod1['description'], $carrierDescriptionForTest);

        $dataMethod2 = $result[1];
        $this->assertArrayHasKey('code', $dataMethod2);
        $this->assertEquals('testcarriercode_test2methodcode', $dataMethod2['code']);
        $this->assertArrayHasKey('description', $dataMethod2);
        $this->assertNotEmpty($dataMethod2['description']);
        $this->assertEquals($dataMethod2['description'], $methodDescriptionForTest);

        $this->assertNotEquals($dataMethod1['description'], $dataMethod2['description']);
    }
}
