<?php
/**
 * Anowave Magento 2 Google Tag Manager Enhanced Ecommerce (UA) Tracking
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Anowave license that is
 * available through the world-wide-web at this URL:
 * https://www.anowave.com/license-agreement/
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category 	Anowave
 * @package 	Anowave_Ec
 * @copyright 	Copyright (c) 2025 Anowave (https://www.anowave.com/)
 * @license  	https://www.anowave.com/license-agreement/
 */
namespace Anowave\Ec\Model;

class Curl implements CurlInterface
{
    /**
     * Access token
     *
     * @var string
     */
    private $access_token;
    
    /**
     * @var CurlHandle
     */
    private $curl;
    
    /**
     * Default headers
     *
     * @var array
     */
    private $headers = [];
    
    /**
     * Constructor
     *
     * @param string $access_token
     */
    public function __construct()
    {
        /**
         * @var \CurlHandle $curl
         */
        $this->curl = curl_init();

        /**
         * Set default options
         */
        $this->reset();
    }

    /**
     * Request
     *
     * @return array
     */
    public function request() : array
    {
        $headers = array_map(function($header, $value)
        {
            return join(chr(58), [$header,$value]);
            
        }, array_keys($this->headers), array_values($this->headers));
        
        /**
         * Add headers
         */
        curl_setopt($this->curl, CURLOPT_HTTPHEADER, $headers);
        
        /**
         * Execute response
         *
         * @var Ambiguous $response
         */
        $response = curl_exec($this->curl);
        
        if (!curl_error($this->curl))
        {
            if ($response)
            {
                $result = json_decode($response, true);

                if (json_last_error() === JSON_ERROR_NONE) 
                {
                    return $result;
                }
                else 
                {
                    return 
                    [
                        'error' => 'Response is not a valid JSON object',
                        'print' => print_r($response, true)
                    ];
                }
            }
        }
        else
        {
            return ['curl_error' => curl_error($this->curl)];
        }
        
        return [];
    }
    
    /**
     * Get request
     *
     * @param string $resource
     * @param array $data
     * @return array
     */
    public function get(string $resource = '', array $data = []) : array
    {
        return $this->reset()->query($resource)->request();
    }
    
    /**
     * Post request
     *
     * @param string $resource
     * @param array $data
     * @return array
     */
    public function post(string $resource = '', array $data = []) : array
    {
        return $this->reset
        (
            [
                CURLOPT_POST            => 1,
                CURLOPT_POSTFIELDS      => http_build_query($data)
            ]
        )
        ->query($resource)->request();
    }

    public function delete(string $resource = '', array $data = []) : array
    {
        return $this->reset
        (
            [
                CURLOPT_CUSTOMREQUEST => 'DELETE'
            ]
        )->query($resource)->request();
    } 
 
    /**
     * Set query resource
     *
     * @param string $resource
     * @return \Curl
     */
    public function query(string $resource = '')
    {
        return $this->reset
        (
            [
                CURLOPT_URL            => $resource
            ]
        );
    }

        /**
     * Reset curl 
     * 
     * @return [type]
     */
    private function reset(array $options = [])
    {
        $options = $this->getOptions() + $options;

        curl_reset($this->curl);

        curl_setopt_array($this->curl, $options);

        return $this;
    }

    /**
     * Get curl options
     * 
     * @return array
     */
    private function getOptions() : array 
    {
        return 
        [
            CURLOPT_HEADER         => 0,
            CURLOPT_RETURNTRANSFER => 1,
            CURLOPT_FOLLOWLOCATION => 1,
            CURLOPT_SSL_VERIFYPEER => 0,
            CURLOPT_MAXREDIRS 	   => 0,
            CURLOPT_CONNECTTIMEOUT => 20,
            CURLOPT_TIMEOUT		   => 20,
            CURLOPT_SSLVERSION     => 6
        ];
    }

    /**
     * Add header
     * 
     * @param array $header
     * 
     * @return [type]
     */
    public function addHeader(string $header = '', string $value = '')
    {
        if ($header && $value)
        {
            $this->headers[$header] = $value;
        }

        return $this;
    }

    /**
     * Get headers
     * 
     * @return array
     */
    public function getHeaders() : array
    {
        return $this->headers;
    }
}