<?php
/**
 * Mageplaza
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the mageplaza.com license that is
 * available through the world-wide-web at this URL:
 * https://www.mageplaza.com/LICENSE.txt
 *
 * DISCLAIMER
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Mageplaza
 * @package     Mageplaza_CustomForm
 * @copyright   Copyright (c) Mageplaza (https://www.mageplaza.com/)
 * @license     https://www.mageplaza.com/LICENSE.txt
 */

namespace Mageplaza\CustomForm\Model;

use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Filesystem;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\Mail\EmailMessage;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\Session\SessionManagerInterface;
use Magento\Store\Model\Store;
use Mageplaza\CustomForm\Helper\Data;
use Symfony\Component\Mime\Part\DataPart;
use Symfony\Component\Mime\Part\TextPart;
use Symfony\Component\Mime\Part\Multipart\MixedPart;

/**
 * Class MailEvent
 * Mageplaza\CustomForm\Model
 */
class MailEvent
{
    /**
     * @var array
     */
    const MIME_TYPES = [
        'txt'  => 'text/plain',
        'pdf'  => 'application/pdf',
        'doc'  => 'application/msword',
        'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    ];

    /**
     * @var array
     */
    private $parts = [];

    /**
     * @var Mail
     */
    private $mail;

    /**
     * @var Data
     */
    private $dataHelper;

    /**
     * @var Filesystem
     */
    private $filesystem;

    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var SessionManagerInterface
     */
    private $coreSession;

    /**
     * @var File
     */
    private $driverFile;

    /**
     * MailEvent constructor.
     *
     * @param Mail $mail
     * @param Data $dataHelper
     * @param Filesystem $filesystem
     * @param ObjectManagerInterface $objectManager
     * @param SessionManagerInterface $coreSession
     * @param File $driverFile
     */
    public function __construct(
        Mail $mail,
        Data $dataHelper,
        Filesystem $filesystem,
        ObjectManagerInterface $objectManager,
        SessionManagerInterface $coreSession,
        File $driverFile
    ) {
        $this->mail          = $mail;
        $this->dataHelper    = $dataHelper;
        $this->filesystem    = $filesystem;
        $this->objectManager = $objectManager;
        $this->coreSession   = $coreSession;
        $this->driverFile    = $driverFile;
    }

    /**
     * @param EmailMessage $message
     *
     * @throws FileSystemException
     */
    public function dispatch($message)
    {
        $templateVars = $this->mail->getTemplateVars();
        if (!$templateVars) {
            return;
        }

        /** @var Store|null $store */
        $store   = isset($templateVars['store']) ? $templateVars['store'] : null;
        $storeId = $store ? $store->getId() : null;

        if (!$this->dataHelper->isEnabled($storeId)) {
            return;
        }

        // Handle file attachments
        if (!empty($templateVars['attachedFiles'])) {
            foreach ($templateVars['attachedFiles'] as $file) {
                $attachment = $this->setAttachedFile($file);
                if ($attachment) {
                    $this->parts[] = $attachment;
                }
            }
        }

        // Attach email body with possible attachments
        $this->setBodyAttachment($message);

        // reset template vars
        $this->mail->setTemplateVars([]);
    }

    /**
     * @param array $file
     *
     * @return DataPart|null
     * @throws FileSystemException
     */
    private function setAttachedFile($file)
    {
        [$content, $mimeType] = $this->getInformationFile($file);

        if (empty($content)) {
            return null;
        }

        return new DataPart($content, $file['name'], $mimeType);
    }

    /**
     * @param array $file
     *
     * @return array
     * @throws FileSystemException
     */
    private function getInformationFile($file)
    {
        $filePath = $file['value'];
        $content  = $this->driverFile->fileGetContents($filePath);
        $ext      = (string) substr($filePath, strrpos($filePath, '.') + 1);
        $mime     = isset(self::MIME_TYPES[$ext]) ? self::MIME_TYPES[$ext] : 'application/octet-stream';

        return [$content, $mime];
    }

    /**
     * @param EmailMessage $message
     */
    private function setBodyAttachment(\Magento\Framework\Mail\EmailMessage $message)
    {
        /** @var \Symfony\Component\Mime\Email $email */
        $email = $message->getSymfonyMessage();

        $body = $email->getBody();

        // If body already exists as TextPart (html/text), keep it
        if ($body instanceof \Symfony\Component\Mime\Part\TextPart) {
            $textPart = $body;
        } else {
            // fallback: wrap as HTML text
            $textPart = new \Symfony\Component\Mime\Part\TextPart((string)$body, 'utf-8', 'html');
        }

        if (!empty($this->parts)) {
            $email->setBody(new \Symfony\Component\Mime\Part\Multipart\MixedPart($textPart, ...$this->parts));
        } else {
            $email->setBody($textPart);
        }
    }
}