<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */
declare(strict_types = 1);

namespace MageWorx\OptionDependency\Model;

use Magento\Catalog\Api\Data\ProductCustomOptionInterface;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Framework\DataObject;
use MageWorx\OptionBase\Helper\Data as BaseHelper;
use MageWorx\OptionBase\Model\HiddenDependents as HiddenDependentsStorage;

class HiddenDependents
{
    protected BaseHelper              $baseHelper;
    protected HiddenDependentsStorage $hiddenDependentsStorage;

    protected array $data                                   = [];
    protected array $hiddenValues                           = [];
    protected array $hiddenOptions                          = [];
    protected array $selectedValues                         = [];
    protected array $optionToValuesMap                      = [];
    protected array $valueToOptionMap                       = [];
    protected array $dependencyRules                        = [];
    protected bool  $hasConfigureQuoteItemsHiddenDependents = false;

    public function __construct(
        BaseHelper              $baseHelper,
        HiddenDependentsStorage $hiddenDependentsStorage
    ) {
        $this->baseHelper              = $baseHelper;
        $this->hiddenDependentsStorage = $hiddenDependentsStorage;
    }

    /**
     * Get hidden dependents
     *
     * @param array $options
     * @param array $dependencyRules
     * @return array
     */
    public function getHiddenDependents(array $options, array $dependencyRules): array
    {
        // Initialize result data
        $this->data = [
            'hidden_options'     => [],
            'hidden_values'      => [],
            'preselected_values' => []
        ];

        if (empty($options)) {
            return $this->data;
        }

        $this->selectedValues    = [];
        $this->optionToValuesMap = [];
        $this->valueToOptionMap  = [];
        $this->dependencyRules   = $dependencyRules;

        $this->collectOptionToValuesMap($options);
        $this->processDependencyRules();

        $isDefaults = $this->getIsDefaults($options);
        $this->processIsDefaults($isDefaults, $options);

        $this->data = [
            'hidden_options'     => array_values($this->hiddenOptions),
            'hidden_values'      => array_values($this->hiddenValues),
            'preselected_values' => $this->getPreparedSelectedValues()
        ];

        return $this->data;
    }

    /**
     * Calculate hidden dependents for ShareableLink or GraphQL's "dependencyState" query
     *
     * @param ProductInterface $product
     * @param array $preselectedValues
     *
     * @return void
     */
    public function calculateHiddenDependents($product, $preselectedValues = []): void
    {
        if (
            !is_a($product, ProductInterface::class) ||
            !is_array($preselectedValues)
        ) {
            return;
        }

        $this->data = [
            'hidden_options'     => [],
            'hidden_values'      => [],
            'preselected_values' => []
        ];

        $options = $this->getOptionsAsArray($product->getOptions());
        if (empty($options)) {
            return;
        }

        $dependencyRulesJson = $product->getDependencyRules();
        try {
            $dependencyRules = $this->baseHelper->jsonDecode($dependencyRulesJson);
        } catch (\Exception $exception) {
            $dependencyRules = [];
        }

        $this->optionToValuesMap = [];
        $this->valueToOptionMap  = [];
        $previousHiddenValues    = [];

        $this->dependencyRules = $dependencyRules;
        $this->selectedValues  = $preselectedValues;

        $this->collectOptionToValuesMap($options);

        $i = 0;
        while ($i < 10000) {
            $this->processDependencyRules();

            $isDefaults = $this->getIsDefaults($options, $this->selectedValues);
            $this->processIsDefaults($isDefaults, $options);

            $hiddenValues = array_values($this->hiddenValues);

            $this->data = [
                'hidden_options'     => array_values($this->hiddenOptions),
                'hidden_values'      => $hiddenValues,
                'preselected_values' => $this->getPreparedSelectedValues()
            ];

            if ($hiddenValues == $previousHiddenValues) {
                break;
            }

            $this->selectedValues = $this->removeHiddenValuesFromSelected($hiddenValues);
            $previousHiddenValues = $hiddenValues;

            $i++;
        }

        $this->hasConfigureQuoteItemsHiddenDependents = true;
        $this->hiddenDependentsStorage->setQuoteItemsHiddenDependents($this->data);
    }

    /**
     * Calculate configure quote items hidden dependents
     *
     * @param ProductInterface $product
     * @param DataObject $buyRequest
     *
     * @return void
     */
    public function calculateConfigureQuoteItemsHiddenDependents($product, $buyRequest): void
    {
        if (
            $this->hasConfigureQuoteItemsHiddenDependents ||
            !is_a($product, ProductInterface::class) ||
            !is_a($buyRequest, DataObject::class)
        ) {
            return;
        }

        $this->data = [
            'hidden_options'     => [],
            'hidden_values'      => [],
            'preselected_values' => []
        ];

        $options = $this->getOptionsAsArray($product->getOptions());
        if (empty($options)) {
            return;
        }

        $dependencyRulesJson = $product->getDependencyRules();
        try {
            $dependencyRules = $this->baseHelper->jsonDecode($dependencyRulesJson);
        } catch (\Exception $exception) {
            $dependencyRules = [];
        }

        $this->selectedValues    = [];
        $this->optionToValuesMap = [];
        $this->valueToOptionMap  = [];
        $this->dependencyRules   = $dependencyRules;

        $this->collectSelectedValuesFromBuyRequest($buyRequest);
        $this->collectOptionToValuesMap($options);
        $this->processDependencyRules();

        $this->data = [
            'hidden_options'     => array_values($this->hiddenOptions),
            'hidden_values'      => array_values($this->hiddenValues),
            'preselected_values' => $this->getPreparedSelectedValues()
        ];

        $this->hasConfigureQuoteItemsHiddenDependents = true;
        $this->hiddenDependentsStorage->setQuoteItemsHiddenDependents($this->data);
    }

    /**
     * Get value to option map from option's array
     *
     * @param array $options
     * @return array
     */
    public function getValueOptionMaps($options)
    {
        $this->optionToValuesMap = [];
        $this->valueToOptionMap  = [];
        if (is_array($options)) {
            $this->collectOptionToValuesMap($options);
        }
        return [
            'valueToOption' => $this->valueToOptionMap,
            'optionToValue' => $this->optionToValuesMap
        ];
    }

    /**
     * Convert option's objects to array and retrieve it.
     *
     * @param array|null $options
     * @return array
     */
    public function getOptionsAsArray(?array $options): array
    {
        if (!$options) {
            $options = [];
        }

        $results = [];

        foreach ($options as $option) {
            if (!is_object($option)) {
                continue;
            }
            $result              = [];
            $result['option_id'] = $option->getOptionId();
            $result['type']      = $option->getType();
            $result['is_hidden'] = $option->getData('is_hidden');

            if ($this->baseHelper->isSelectableOption($option->getType()) && $option->getValues()) {
                foreach ($option->getValues() as $value) {
                    $i = $value->getOptionTypeId();
                    foreach ($value->getData() as $valueKey => $valueDatum) {
                        $result['values'][$i][$valueKey] = $valueDatum;
                    }
                }
            } else {
                foreach ($option->getData() as $optionKey => $optionDatum) {
                    $result[$optionKey] = $optionDatum;
                }
                $result['values'] = null;
            }

            $results[$option->getOptionId()] = $result;
        }

        return $results;
    }

    /**
     * Getter for HiddenDependents::hasConfigureQuoteItemsHiddenDependents
     *
     * @return array
     */
    public function getConfigureQuoteItemsHiddenDependents(): array
    {
        if ($this->hasConfigureQuoteItemsHiddenDependents) {
            return $this->data;
        }

        return [];
    }

    /**
     * Collect selected values from quote item's buyRequest
     *
     * @param DataObject $buyRequest
     * @return void
     */
    protected function collectSelectedValuesFromBuyRequest(DataObject $buyRequest): void
    {
        $selectedOptions = $buyRequest->getData('options');
        if (!$selectedOptions || !is_array($selectedOptions)) {
            return;
        }
        foreach ($selectedOptions as $selectedValues) {
            if (empty($selectedValues)) {
                continue;
            }
            if (is_array($selectedValues)) {
                foreach ($selectedValues as $selectedValue) {
                    $this->selectedValues[] = $selectedValue;
                }
            } else {
                $this->selectedValues[] = $selectedValues;
            }
        }
    }

    /**
     * Get product collection using selected product IDs
     *
     * @param array $options
     * @return array
     */
    protected function getIsDefaults(array $options, array $preselectedValues = []): array
    {
        $isDefaults = [];
        foreach ($options as $option) {
            if (empty($option['values'])) {
                continue;
            }

            // Do not process disabled option values
            if (isset($option['disabled']) && (bool)$option['disabled'] === true) {
                continue;
            }

            /**
             * Only one value can be selected for radio or drop down type options
             */
            if (
                !empty($option['type']) &&
                $this->isOneChoiceOptionType($option['type']) &&
                $this->isThereConflict($option['values'], $preselectedValues)
            ) {
                continue;
            }

            foreach ($option['values'] as $value) {
                if (empty($value['is_default'])) {
                    continue;
                }
                $isDefaults[] = [
                    'option_type_id' => $value['option_type_id'],
                    'is_default'     => $value['is_default']
                ];
            }
        }
        return $isDefaults;
    }

    /**
     * Checks for conflict between selected values and default values
     * @return void
     */
    protected function isThereConflict(array $values, array $preselectedValues): bool
    {
        if (empty($preselectedValues)) {
            return false;
        }
        foreach ($values as $value) {
            if (isset($value['option_type_id']) && in_array($value['option_type_id'], $preselectedValues)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Process all dependency rules.
     *
     * This method delegates to either the “and” or “or” processors.
     */
    protected function collectOptionToValuesMap(array $options): void
    {
        foreach ($options as $option) {
            if (!isset($option['option_id'])
                || empty($option['values'])
                || !empty($option['is_disabled'])
            ) {
                continue;
            }
            $valueIds = [];
            foreach ($option['values'] as $value) {
                if (!isset($value['option_type_id'])
                    || !empty($value['is_disabled'])
                ) {
                    continue;
                }
                $valueIds[]                                       = $value['option_type_id'];
                $this->valueToOptionMap[$value['option_type_id']] = $option['option_id'];
            }
            $this->optionToValuesMap[$option['option_id']] = $valueIds;
        }
    }

    protected function isOneChoiceOptionType(string $type): bool
    {
        return in_array($type, [ProductCustomOptionInterface::OPTION_TYPE_RADIO, ProductCustomOptionInterface::OPTION_TYPE_DROP_DOWN]);
    }

    protected function processDependencyRules(): void
    {
        // Reset hidden values/options for a fresh pass.
        $this->hiddenValues  = [];
        $this->hiddenOptions = [];
        foreach ($this->dependencyRules as $dependencyRule) {
            if (!$this->isValidRule($dependencyRule)) {
                continue;
            }

            if ($dependencyRule['condition_type'] === 'and') {
                $this->processDependencyAndRules($dependencyRule);
            } else { // default to 'or'
                $this->processDependencyOrRules($dependencyRule);
            }
        }
        $this->hideOptionIfAllValuesHidden();
    }

    /**
     * Process dependency rules with OR logic.
     *
     * For OR–rules, if any condition fails then the rule is skipped.
     *
     * @param array $dependencyRule
     */
    protected function processDependencyOrRules(array $dependencyRule): void
    {
        if (empty($dependencyRule['conditions'])) {
            // No conditions means nothing to do.
            return;
        }

        // Loop over each condition. For a '!eq' condition,
        // if any of its values is selected, then we do not hide.
        foreach ($dependencyRule['conditions'] as $condition) {
            $values = $this->getConditionValues($condition);
            if ($condition['type'] === '!eq' && $this->hasSelectedValue($values)) {
                // One condition failed (i.e. a value that should not be selected was selected)
                return;
            }
            // Note: The 'eq' type is not implemented yet.
        }

        // All conditions passed, so perform the hide action.
        $this->addHiddenValuesByRule($dependencyRule);
    }

    /**
     * Process dependency rules with AND logic.
     *
     * For AND–rules, if at least one condition finds that not all required values are selected,
     * the hide action is performed.
     *
     * @param array $dependencyRule
     */
    protected function processDependencyAndRules(array $dependencyRule): void
    {
        if (empty($dependencyRule['conditions'])) {
            return;
        }

        foreach ($dependencyRule['conditions'] as $condition) {
            $values = $this->getConditionValues($condition);
            if ($condition['type'] === '!eq' && !$this->allValuesSelected($values)) {
                $this->addHiddenValuesByRule($dependencyRule);
                return;
            }
            // 'eq' type is not implemented.
        }
    }

    /**
     * Get the option values to use for checking a condition.
     *
     * If the condition does not explicitly specify 'values' and an 'id' is provided,
     * then use the mapping created from options.
     *
     * @param array $condition
     * @return array
     */
    protected function getConditionValues(array $condition): array
    {
        $values = $condition['values'] ?? [];
        if (empty($values) && !empty($condition['id'])) {
            $values = $this->optionToValuesMap[$condition['id']] ?? [];
        }
        return $values;
    }

    /**
     * Check if at least one value in $values is present in the selected values.
     *
     * @param array $values
     * @return bool
     */
    protected function hasSelectedValue(array $values): bool
    {
        foreach ($values as $value) {
            if (in_array($value, $this->selectedValues, true)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Check if all of the given values are selected.
     *
     * @param array $values
     * @return bool
     */
    protected function allValuesSelected(array $values): bool
    {
        foreach ($values as $value) {
            if (!in_array($value, $this->selectedValues, true)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Add hidden values and options based on the dependency rule’s hide action.
     *
     * @param array $dependencyRule
     */
    protected function addHiddenValuesByRule(array $dependencyRule): void
    {
        foreach ($dependencyRule['actions']['hide'] as $hideItem) {
            // If specific values are provided, hide each value.
            if (!empty($hideItem['values']) && is_array($hideItem['values'])) {
                foreach ($hideItem['values'] as $hideValueId) {
                    if ($hideValueId) {
                        $this->hiddenValues[(int)$hideValueId] = (int)$hideValueId;
                    }
                }
            } else {
                // Otherwise, hide the whole option (and all its values).
                if (empty($hideItem['id'])) {
                    continue;
                }
                $this->hiddenOptions[(int)$hideItem['id']] = (int)$hideItem['id'];
                $optionValues                              = $this->optionToValuesMap[$hideItem['id']] ?? [];
                foreach ($optionValues as $hideValueId) {
                    if ($hideValueId) {
                        $this->hiddenValues[(int)$hideValueId] = (int)$hideValueId;
                    }
                }
            }
        }
    }

    /**
     * If all the option’s values are hidden, hide the option.
     */
    protected function hideOptionIfAllValuesHidden(): void
    {
        foreach ($this->optionToValuesMap as $optionId => $valueIds) {
            if (empty($valueIds)) {
                continue;
            }
            $allHidden = true;
            foreach ($valueIds as $valueId) {
                // Special check: if a value starts with "o" (as a string) we do not consider it hidden.
                if (!in_array($valueId, $this->hiddenValues, true) || ((string)$valueId)[0]) {
                    $allHidden = false;
                    break;
                }
            }
            if ($allHidden) {
                $this->hiddenOptions[(int)$optionId] = (int)$optionId;
            }
        }
    }

    /**
     * Remove hidden values from the current selected values.
     *
     * @param array $hiddenValues
     * @return array
     */
    protected function removeHiddenValuesFromSelected(array $hiddenValues): array
    {
        // Using array_diff to simplify filtering.
        return array_values(array_diff($this->selectedValues, $hiddenValues));
    }

    /**
     * Process default selections recursively.
     *
     * If a default value can be preselected, add it, then reprocess dependency rules.
     *
     * @param array $isDefaults
     * @param array $options
     */
    protected function processIsDefaults(array $isDefaults, array $options): void
    {
        if (empty($isDefaults)) {
            return;
        }

        $newSelectedAdded = false;
        foreach ($isDefaults as $default) {
            if (!isset($default['option_type_id'])) {
                continue;
            }
            $optionTypeId = (int)$default['option_type_id'];
            foreach ($options as $option) {
                if (!empty($option['is_hidden']) && (int)$option['is_hidden'] !== 0) {
                    continue;
                }
                if (!empty($option['values'])) {
                    foreach ($option['values'] as $value) {
                        if ((int)$value['option_type_id'] === $optionTypeId &&
                            (!isset($value['qty']) || (int)$value['qty'] === 0) &&
                            isset($value['manage_stock']) && (int)$value['manage_stock'] !== 0
                        ) {
                            $this->hiddenValues[$optionTypeId] = $optionTypeId;
                        }
                    }
                }
            }

            if ($this->cannotBePreselectedValue($default, $optionTypeId)) {
                continue;
            }
            if (!in_array($default['option_type_id'], $this->selectedValues, true)) {
                $this->selectedValues[] = $default['option_type_id'];
                $newSelectedAdded       = true;
            }
        }

        if ($newSelectedAdded) {
            $this->processDependencyRules();
            // Re-run defaults in case the new selections cause further defaults to apply.
            $this->processIsDefaults($isDefaults, $options);
        }
    }

    /**
     * Check whether a default value can be preselected.
     *
     * @param array $default
     * @param int $optionTypeId
     * @return bool
     */
    protected function cannotBePreselectedValue(array $default, int $optionTypeId): bool
    {
        return empty($default['is_default'])
               || in_array($default['option_type_id'], $this->selectedValues, true)
               || in_array($optionTypeId, $this->hiddenValues, true)
               || empty($this->valueToOptionMap[$default['option_type_id']]);
    }

    protected function getPreparedSelectedValues(): array
    {
        $data = [];
        foreach ($this->selectedValues as $selectedValue) {
            if (!isset($this->valueToOptionMap[$selectedValue])) {
                continue;
            }
            $data[$this->valueToOptionMap[$selectedValue]][] = (int)$selectedValue;
        }

        return $data;
    }

    /**
     * Check that the dependency rule has the required keys.
     *
     * @param mixed $dependencyRule
     * @return bool
     */
    public function isValidRule($dependencyRule): bool
    {
        return is_array($dependencyRule)
               && isset($dependencyRule['condition_type'], $dependencyRule['conditions'], $dependencyRule['actions']);
    }
}
