<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingCalculatorBase\Test\Unit\Model;

use MageWorx\ShippingCalculatorBase\Model\GeneralAddressResolver;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Customer\Model\Session as CustomerSession;
use MageWorx\ShippingCalculatorBase\Helper\Data as Helper;
use Magento\Customer\Model\Customer;
use Magento\Customer\Model\Address as CustomerAddress;

class GeneralAddressResolverTest extends \PHPUnit\Framework\TestCase
{
    /**
     * Expected parameters config
     */
    protected $expectedCountrySession = 'BY';
    protected $expectedCountryAddress = 'UK';
    protected $expectedRegionIdSession = '1';
    protected $expectedRegionIdAddress = '256';
    protected $expectedRegionAddress = 'Minsk';
    protected $expectedPostcodeSession = '220034';
    protected $expectedPostcodeAddress = '60343';

    /**
     * @var GeneralAddressResolver
     */
    protected $generalAddressResolverClass;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    protected $sessionMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    protected $checkoutSessionMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    protected $helperMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    protected $regionCollectionMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    protected $customerMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    protected $customerAddressMock;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    protected $regionCollectionFactoryMock;

    public function setUp(): void
    {
        $objectManager = new \Magento\Framework\TestFramework\Unit\Helper\ObjectManager($this);
        $this->regionCollectionMock = $this->getMockBuilder(
            \Magento\Directory\Model\ResourceModel\Region\Collection::class
        )
                                           ->disableOriginalConstructor()
                                           ->getMock();
        $this->regionCollectionMock->method('setPageSize')
                                   ->willReturnSelf();
        $this->regionCollectionMock->method('getFirstItem')
                                   ->willReturn(
                                       $objectManager->getObject(
                                           \Magento\Framework\DataObject::class,
                                           [
                                               'data' => [
                                                   'id' => 1 // Skip validation like item in collection exists
                                               ]
                                           ]
                                       )
                                   );
        $this->regionCollectionFactoryMock = $this->getMockBuilder(
            \Magento\Directory\Model\ResourceModel\Region\CollectionFactory::class
        )->disableOriginalConstructor()
                                                  ->getMock();
        $this->regionCollectionFactoryMock->method('create')
                                          ->willReturn($this->regionCollectionMock);

        $this->sessionMock = $this->getMockBuilder(CustomerSession::class)
                                  ->disableOriginalConstructor()
                                  ->setMethods(
                                      ['getCountryId', 'getCustomer', 'getRegionId', 'getRegion', 'getPostcode']
                                  )
                                  ->getMock();

        $this->customerMock = $this->getMockBuilder(Customer::class)
                                   ->disableOriginalConstructor()
                                   ->getMock();

        $this->customerAddressMock = $this->getMockBuilder(
            CustomerAddress::class
        )
                                          ->disableOriginalConstructor()
                                          ->setMethods(
                                              ['getCountryId', 'getCountry', 'getRegionId', 'getRegion', 'getPostcode']
                                          )
                                          ->getMock();

        $this->sessionMock->expects($this->any())->method('getCustomer')->willReturn($this->customerMock);

        $this->checkoutSessionMock = $this->getMockBuilder(CheckoutSession::class)
                                          ->disableOriginalConstructor()
                                          ->getMock();

        $this->helperMock = $this->getMockBuilder(Helper::class)
                                 ->disableOriginalConstructor()
                                 ->getMock();

        $this->generalAddressResolverClass = $objectManager->getObject(
            GeneralAddressResolver::class,
            [
                'regionCollectionFactory' => $this->regionCollectionFactoryMock,
                'checkoutSession'         => $this->checkoutSessionMock,
                'customerSession'         => $this->sessionMock,
                'helper'                  => $this->helperMock
            ]
        );
    }

    // Country ID

    /**
     * Testing country id detection using data stored in session
     */
    public function testGetCountryIdFromSession()
    {
        $this->sessionMock->expects($this->atLeastOnce())->method('getCountryId')->willReturn(
            $this->expectedCountrySession
        );
        $this->assertEquals($this->expectedCountrySession, $this->generalAddressResolverClass->getCountryId());
    }

    /**
     * Testing country id detection using data from customer default address
     */
    public function testGetCountryIdFromCustomerAddress()
    {
        $this->sessionMock->expects($this->atLeastOnce())->method('getCountryId')->willReturn(null);
        $this->customerAddressMock->expects($this->atLeastOnce())->method('getCountry')->willReturn(
            $this->expectedCountrySession
        );
        $this->customerMock->method('getDefaultShippingAddress')->willReturn($this->customerAddressMock);
        $this->sessionMock->method('getCustomer')->willReturn($this->customerMock);

        $this->assertEquals($this->expectedCountrySession, $this->generalAddressResolverClass->getCountryId());
    }

    // Region and Region ID

    /**
     * Testing region id stored in the session
     */
    public function testGetRegionIdSession()
    {
        $this->sessionMock->expects($this->atLeastOnce())->method('getRegionId')->willReturn(
            $this->expectedRegionIdSession
        );
        $this->assertEquals($this->expectedRegionIdSession, $this->generalAddressResolverClass->getRegionId());
    }

    /**
     * Testing region id detection using data from customer default address having Region ID
     */
    public function testGetRegionIdFromCustomerAddress()
    {
        $this->sessionMock->expects($this->atLeastOnce())->method('getRegionId')->willReturn(null);
        $this->customerAddressMock->expects($this->atLeastOnce())->method('getRegionId')->willReturn(
            $this->expectedRegionIdAddress
        );
        $this->customerMock->method('getDefaultShippingAddress')->willReturn($this->customerAddressMock);
        $this->sessionMock->method('getCustomer')->willReturn($this->customerMock);

        $this->assertEquals($this->expectedRegionIdAddress, $this->generalAddressResolverClass->getRegionId());
    }

    /**
     * Testing region detection using data from customer default address without Region ID
     */
    public function testGetRegionFromCustomerAddress()
    {
        $this->sessionMock->expects($this->atLeastOnce())->method('getRegionId')->willReturn(null);
        $this->sessionMock->expects($this->any())->method('getRegion')->willReturn(null);
        $this->customerAddressMock->expects($this->atLeastOnce())->method('getRegionId')->willReturn(null);
        $this->customerAddressMock->expects($this->atLeastOnce())->method('getRegion')->willReturn(
            $this->expectedRegionAddress
        );
        $this->customerMock->method('getDefaultShippingAddress')->willReturn($this->customerAddressMock);
        $this->sessionMock->method('getCustomer')->willReturn($this->customerMock);

        // The getRegionId() method will process both cases: region_id and region
        // (in case the region_id is not available), this is not a mistake!
        $this->assertEquals($this->expectedRegionAddress, $this->generalAddressResolverClass->getRegionId());
    }

    // Postcode

    /**
     * Testing postcode obtained from session
     */
    public function testGetPostcodeSession()
    {
        $this->sessionMock->expects($this->atLeastOnce())->method('getPostcode')->willReturn(
            $this->expectedPostcodeSession
        );
        $this->assertEquals($this->expectedPostcodeSession, $this->generalAddressResolverClass->getZipCode());
    }

    /**
     * Testing postcode detection using data from customer default address
     */
    public function testGetPostcodeFromCustomerAddress()
    {
        $this->sessionMock->expects($this->atLeastOnce())->method('getPostcode')->willReturn(null);
        $this->customerAddressMock->expects($this->atLeastOnce())
                                  ->method('getPostcode')
                                  ->willReturn($this->expectedPostcodeAddress);
        $this->customerMock->method('getDefaultShippingAddress')->willReturn($this->customerAddressMock);
        $this->sessionMock->method('getCustomer')->willReturn($this->customerMock);

        $this->assertEquals($this->expectedPostcodeAddress, $this->generalAddressResolverClass->getZipCode());
    }
}
