<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\ShippingCalculatorBase\Controller\Calculator;

use Exception;
use Magento\Customer\Model\Session;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\State;
use Magento\Framework\Controller\Result\Json as ResultJson;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Checkout\Model\Session as CheckoutSession;
use MageWorx\ShippingCalculatorBase\Api\EstimateShippingMethodsInterface;
use MageWorx\ShippingCalculatorBase\Helper\Data as Helper;
use MageWorx\ShippingCalculatorBase\Plugin\ShippingRules\QuoteSubstitution as QuoteSubstitutionPlugin;
use Psr\Log\LoggerInterface;

/**
 * Class Refresh
 */
class Refresh extends Action
{
    /**
     * @var Session
     */
    protected $session;

    /**
     * @var JsonFactory
     */
    protected $jsonResultFactory;

    /**
     * @var CheckoutSession
     */
    protected $checkoutSession;

    /**
     * @var State
     */
    protected $appState;

    /**
     * @var EstimateShippingMethodsInterface
     */
    private $estimateShippingMethods;

    /**
     * @var QuoteSubstitutionPlugin
     */
    private $quoteSubstitutionPlugin;

    /**
     * @var Helper
     */
    private $helper;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * Refresh constructor.
     *
     * @param Context $context
     * @param Session $customerSession
     * @param JsonFactory $resultJsonFactory
     * @param CheckoutSession $checkoutSession
     * @param State $appState
     * @param EstimateShippingMethodsInterface $estimateShippingMethods
     * @param QuoteSubstitutionPlugin $quoteSubstitutionPlugin
     * @param Helper $helper
     */
    public function __construct(
        Context $context,
        Session $customerSession,
        JsonFactory $resultJsonFactory,
        CheckoutSession $checkoutSession,
        State $appState,
        EstimateShippingMethodsInterface $estimateShippingMethods,
        QuoteSubstitutionPlugin $quoteSubstitutionPlugin,
        Helper $helper,
        LoggerInterface $logger
    ) {
        $this->session                 = $customerSession;
        $this->jsonResultFactory       = $resultJsonFactory;
        $this->checkoutSession         = $checkoutSession;
        $this->appState                = $appState;
        $this->estimateShippingMethods = $estimateShippingMethods;
        $this->quoteSubstitutionPlugin = $quoteSubstitutionPlugin;
        $this->helper                  = $helper;
        $this->logger                  = $logger;

        parent::__construct($context);
    }

    /**
     * Execute action based on request and return result
     *
     * Note: Request will be added as operation argument in future
     *
     * @return ResultInterface|ResultJson
     */
    public function execute(): ResultJson
    {
        /** @var ResultJson $result */
        $result = $this->jsonResultFactory->create();
        try {
            $shippingAddressData = $this->getRequest()->getParam('address');
            $itemData            = $this->getRequest()->getParam('item');

            if ($shippingAddressData && $itemData) {
                $normalItemData = $this->normalizeItemFormData($itemData);
                $this->saveShippingAddressData($shippingAddressData);
                $this->quoteSubstitutionPlugin->setSubstitutionFlag(true);
                $this->updateAddressDataForTableRates($shippingAddressData);
                $shippingMethods = $this->estimateShippingMethods->estimate($shippingAddressData, $normalItemData);

                $result->setData(
                    [
                        'success'          => true,
                        'time'             => time(),
                        'shipping_methods' => $shippingMethods,
                        'error'            => empty($shippingMethods) ?
                            $this->helper->getCalculatorNotFoundErrorMessage() :
                            false
                    ]
                );
            } else {
                $error = __('Please select Your country');
                $result->setData(
                    [
                        'success'          => false,
                        'time'             => time(),
                        'shipping_methods' => [],
                        'error'            => $error
                    ]
                );
            }
        } catch (LocalizedException $e) {
            $error = $e->getMessage();
            $this->logger->critical($error, [$this->getRequest()->getParams()]);
            $result->setData(
                [
                    'success'          => false,
                    'time'             => time(),
                    'shipping_methods' => [],
                    'error'            => $error,
                    'trace'            => $e->getTraceAsString()
                ]
            );
        } catch (Exception $exception) {
            $this->logger->critical($exception->getMessage(), [$this->getRequest()->getParams()]);
            if ($this->appState->getMode() == State::MODE_DEVELOPER) {
                $error = $exception->getMessage();
            } else {
                $error = __('Something went wrong. Please try again later.');
            }
            $result->setData(
                [
                    'success'          => false,
                    'time'             => time(),
                    'shipping_methods' => [],
                    'error'            => $error,
                    'trace'            => $exception->getTraceAsString()
                ]
            );
        }

        return $result;
    }

    /**
     * Add destination address for the tablerate request calculations
     *
     * @param array $addressData
     */
    private function updateAddressDataForTableRates(array &$addressData = []): void
    {
        if (!empty($addressData['country_id']) && empty($addressData['dest_country_id'])) {
            $addressData['dest_country_id'] = $addressData['country_id'];
        }

        if (!empty($addressData['region_id']) && empty($addressData['dest_region_id'])) {
            $addressData['dest_region_id'] = $addressData['region_id'];
        }

        if (!empty($addressData['region']) && empty($addressData['dest_region'])) {
            $addressData['dest_region'] = $addressData['region'];
        }

        if (!empty($addressData['postcode']) && empty($addressData['dest_postcode'])) {
            $addressData['dest_postcode'] = $addressData['postcode'];
        }

        if (empty($addressData['website_id']) && $this->checkoutSession->getQuote()) {
            $quote = $this->checkoutSession->getQuote();
            $store = $quote->getStore();
            if ($store) {
                $addressData['website_id'] = $store->getWebsiteId();
            }
        }
    }

    /**
     * Save data in session for later use
     *
     * @param array $data
     */
    private function saveShippingAddressData(array $data = []): void
    {
        $countryId = $data['country_id'] ?? '';
        $regionId  = $data['region_id'] ?? '';
        $region    = $data['region'] ?? '';
        $postcode  = $data['postcode'] ?? '';
        $this->session->setCountryId($countryId);
        $this->session->setRegionId($regionId);
        $this->session->setRegion($region);
        $this->session->setPostcode($postcode);
    }

    /**
     * Convert key-value pairs array to associative array
     *
     * @param array $itemData
     * @return array
     */
    private function normalizeItemFormData(array $itemData = []): array
    {
        $normalData = [];
        foreach ($itemData as $value) {
            $name         = $value['name'];
            $value        = $value['value'];
            $subKeyExists = preg_match_all('/\[([\d]+|[\w]{0,})\]/', $name, $matches);
            if ($subKeyExists && isset($matches[1])) {
                $name              = preg_replace('/\[(.+)\]/', '', $name);
                $i                 = 0;
                $count             = count($matches[1]);
                $normalData[$name] = !empty($normalData[$name]) ? $normalData[$name] : [];
                $point             = &$normalData[$name];

                while ($i < $count) {
                    $key = $matches[1][$i];
                    if ($key === '') {
                        $point[] = $value;
                    } else {
                        if (($i + 1) === $count) {
                            $point[$key] = $value;
                        } else {
                            $point[$key] = !empty($point[$key]) ? $point[$key] : [];
                            $point = &$point[$matches[1][$i]];
                        }
                    }
                    $i++;
                }
            } else {
                $normalData[$name] = $value;
            }
        }

        return $normalData;
    }
}
