<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */
declare(strict_types = 1);

namespace MageWorx\OptionDependency\Test\Unit\Model;

use MageWorx\OptionBase\Helper\Data as BaseHelper;
use MageWorx\OptionBase\Model\HiddenDependents as HiddenDependentsStorage;
use MageWorx\OptionDependency\Model\HiddenDependents;
use PHPUnit\Framework\TestCase;

class HiddenDependentsTest extends TestCase
{
    private TestableHiddenDependents $hiddenDependents;

    protected function setUp(): void
    {
        // Create simple dummy dependencies for BaseHelper and HiddenDependentsStorage.
        $baseHelper = $this->createMock(BaseHelper::class);
        // We won’t use any BaseHelper methods in this test so no further configuration is needed.
        $hiddenDependentsStorage = $this->createMock(HiddenDependentsStorage::class);

        $this->hiddenDependents = new TestableHiddenDependents($baseHelper, $hiddenDependentsStorage);
    }

    /**
     * Test that for an OR rule with a '!eq' condition,
     * if none of the condition values is selected, the hide action is performed.
     */
    public function testProcessDependencyOrRules_HidesValuesWhenNoneSelected(): void
    {
        // Set up the state:
        // Assume no values are selected.
        $this->hiddenDependents->setSelectedValues([]);
        // And assume the mapping for option id 1 returns values [101, 102]
        $this->hiddenDependents->setOptionToValuesMap(
            [
                1 => [101, 102]
            ]
        );

        // Create a dependency rule where the condition references option id 1.
        $dependencyRule = [
            'condition_type' => 'or',
            'conditions'     => [
                [
                    'id'     => 1,
                    'type'   => '!eq',
                    'values' => []  // empty means use optionToValuesMap
                ]
            ],
            'actions'        => [
                'hide' => [
                    [
                        'values' => [201, 202]
                    ]
                ]
            ]
        ];

        $this->hiddenDependents->testProcessDependencyOrRules($dependencyRule);

        $hiddenValues = $this->hiddenDependents->getHiddenValues();
        // We expect that the hide action added values 201 and 202.
        $this->assertArrayHasKey(201, $hiddenValues);
        $this->assertArrayHasKey(202, $hiddenValues);
        $this->assertEquals(201, $hiddenValues[201]);
        $this->assertEquals(202, $hiddenValues[202]);
    }

    /**
     * Test that if one of the condition values is selected,
     * the rule is not applied.
     */
    public function testProcessDependencyOrRules_DoesNotHideWhenValueSelected(): void
    {
        // Set up the state: the selected values include one forbidden value.
        $this->hiddenDependents->setSelectedValues([102]);
        $this->hiddenDependents->setOptionToValuesMap(
            [
                1 => [101, 102]
            ]
        );

        $dependencyRule = [
            'condition_type' => 'or',
            'conditions'     => [
                [
                    'id'     => 1,
                    'type'   => '!eq',
                    'values' => []  // will be replaced by [101,102]
                ]
            ],
            'actions'        => [
                'hide' => [
                    [
                        'values' => [201, 202]
                    ]
                ]
            ]
        ];

        $this->hiddenDependents->testProcessDependencyOrRules($dependencyRule);

        $hiddenValues = $this->hiddenDependents->getHiddenValues();
        // Because 102 is selected (and it is one of the values to check), no hiding should occur.
        $this->assertEmpty($hiddenValues);
    }
}
